<?php

namespace App\Console\Commands;

use App\Jobs\ProcesarEnriquecimientoNumerosJob;
use App\Models\EnriquecimientoImportacion;
use App\Models\UsuarioCredito;
use Illuminate\Console\Command;
use Illuminate\Support\Facades\Log;

class ReanudarImportacionesPausadas extends Command
{
    /**
     * The name and signature of the console command.
     *
     * @var string
     */
    protected $signature = 'enriquecimiento:reanudar-importaciones {--user_id=}';

    /**
     * The console command description.
     *
     * @var string
     */
    protected $description = 'Reanuda el procesamiento de importaciones pausadas cuando hay créditos disponibles';

    /**
     * Execute the console command.
     */
    public function handle()
    {
        $userId = $this->option('user_id');

        if ($userId) {
            $this->reanudarParaUsuario($userId);
        } else {
            $this->reanudarParaTodosLosUsuarios();
        }

        return Command::SUCCESS;
    }

    private function reanudarParaUsuario($userId)
    {
        $this->info("Verificando importaciones pausadas para usuario {$userId}...");

        $creditosDisponibles = UsuarioCredito::numerosDisponiblesParaProcesar($userId);

        if ($creditosDisponibles <= 0) {
            $this->warn("Usuario {$userId} no tiene créditos disponibles");
            return;
        }

        $importacionesPausadas = EnriquecimientoImportacion::where('user_id', $userId)
            ->whereIn('estado', ['pausada', 'pendiente'])
            ->orderBy('created_at', 'asc')
            ->get();

        if ($importacionesPausadas->isEmpty()) {
            $this->info("No hay importaciones pausadas para el usuario {$userId}");
            return;
        }

        $this->info("Encontradas {$importacionesPausadas->count()} importaciones pausadas");
        $this->info("Créditos disponibles: {$creditosDisponibles}");

        foreach ($importacionesPausadas as $importacion) {
            if ($creditosDisponibles <= 0) {
                $this->warn("Se agotaron los créditos, pausando reanudación");
                break;
            }

            $numerosPendientes = $importacion->numeros()->where('estado', 'pendiente')->count();

            if ($numerosPendientes > 0) {
                $cantidadAProcesar = min($creditosDisponibles, $numerosPendientes);

                $this->info("Reanudando importación {$importacion->id} - Procesando {$cantidadAProcesar} números");

                ProcesarEnriquecimientoNumerosJob::dispatch($importacion, $cantidadAProcesar);

                $creditosDisponibles -= $cantidadAProcesar;

                Log::info("Importación reanudada automáticamente", [
                    'importacion_id' => $importacion->id,
                    'user_id' => $userId,
                    'numeros_a_procesar' => $cantidadAProcesar
                ]);
            }
        }

        $this->info("Proceso completado para usuario {$userId}");
    }

    private function reanudarParaTodosLosUsuarios()
    {
        $this->info("Verificando importaciones pausadas para todos los usuarios...");

        $importacionesPausadas = EnriquecimientoImportacion::whereIn('estado', ['pausada', 'pendiente'])
            ->with('user')
            ->get()
            ->groupBy('user_id');

        if ($importacionesPausadas->isEmpty()) {
            $this->info("No hay importaciones pausadas");
            return;
        }

        foreach ($importacionesPausadas as $userId => $importaciones) {
            $this->reanudarParaUsuario($userId);
        }

        $this->info("Proceso completado para todos los usuarios");
    }
}
