<?php

namespace App\Http\Controllers\Auth;

use App\Http\Controllers\Controller;
use App\Http\Requests\Auth\LoginRequest;
use Illuminate\Http\RedirectResponse;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Auth;
use Illuminate\Support\Facades\Route;
use Inertia\Inertia;
use Inertia\Response;
use Illuminate\Validation\ValidationException;

class AuthenticatedSessionController extends Controller
{
    /**
     * Display the login view.
     */
    public function create(): Response
    {
        return Inertia::render('Auth/Login', [
            'canResetPassword' => Route::has('password.request'),
            'status' => session('status'),
        ]);
    }

    /**
     * Handle an incoming authentication request.
     */
    public function store(LoginRequest $request)
    {
        // Detectar si hay un mensaje de bloqueo en sesión
        if (session()->has('auth_lockout')) {
            return Inertia::render('Auth/Login', [
                'errors' => [
                    'email' => session('auth_lockout'),
                ],
            ]);
        }

        try {
            $request->authenticate();
            $request->session()->regenerate();

            // Verificar límite de sesiones DESPUÉS de autenticar pero ANTES de redirigir
            /** @var \App\Models\User $user */
            $user = \Illuminate\Support\Facades\Auth::user();
            $currentSessionId = session()->getId();

            // Limpiar sesiones inactivas primero
            \App\Models\UserSession::limpiarSesionesInactivas();

            // Contar sesiones activas excluyendo la sesión actual (que aún no se ha registrado)
            $sesionesActivas = $user->sesionesActivas()->count();
            $limiteSesiones = $user->getLimiteSesiones();

            // Si ya hay sesiones que igualan o superan el límite, mostrar página de límite
            // porque al agregar la sesión actual se excedería
            if ($sesionesActivas >= $limiteSesiones) {
                // Obtener información de otras sesiones
                $otrasSesiones = $user->sesionesActivas()
                    ->select('id', 'device_name', 'ip_address', 'last_activity', 'location')
                    ->get()
                    ->map(function ($session) {
                        return [
                            'id' => $session->id,
                            'device' => $session->device_name,
                            'ip' => $session->ip_address,
                            'location' => $session->location,
                            'last_activity' => $session->last_activity->diffForHumans(),
                        ];
                    });

                // Renderizar página de manejo de límite de sesiones
                return \Inertia\Inertia::render('Auth/SessionLimit', [
                    'user_plan' => $user->plan,
                    'sessions_limit' => $limiteSesiones,
                    'active_sessions_count' => $sesionesActivas,
                    'other_sessions' => $otrasSesiones,
                    'limit_exceeded' => true
                ]);
            }

            return redirect()->intended(route('dashboard'));
        } catch (ValidationException $e) {
            return Inertia::render('Auth/Login', [
                'errors' => $e->errors(),
            ]);
        }
    }


    /**
     * Destroy an authenticated session.
     */
    public function destroy(Request $request): RedirectResponse
    {
        $user = Auth::user();
        $sessionId = session()->getId();

        // Limpiar la sesión de la tabla user_sessions antes de hacer logout
        if ($user) {
            \App\Models\UserSession::where('user_id', $user->id)
                ->where('session_id', $sessionId)
                ->delete();
        }

        Auth::guard('web')->logout();

        $request->session()->invalidate();

        $request->session()->regenerateToken();

        return redirect('/');
    }

    /**
     * Muestra la página de límite de sesiones cuando se excede por navegación directa
     */
    public function showSessionLimit(Request $request)
    {
        /** @var \App\Models\User $user */
        $user = Auth::user();

        if (!$user) {
            return redirect()->route('login');
        }

        // Obtener datos de sesión guardados por el middleware
        $sessionLimitData = $request->session()->get('session_limit_data');

        // Si no hay datos guardados, verificar estado actual
        if (!$sessionLimitData) {
            \App\Models\UserSession::limpiarSesionesInactivas();

            $currentSessionId = session()->getId();
            $sesionesActivas = $user->sesionesActivas()->count();
            $limiteSesiones = $user->getLimiteSesiones();

            // Si ya no excede el límite, redirigir al dashboard
            if ($sesionesActivas <= $limiteSesiones) {
                return redirect()->route('dashboard');
            }

            // Crear datos para la vista
            $sessionLimitData = [
                'active_sessions' => $sesionesActivas,
                'max_sessions' => $limiteSesiones,
                'other_sessions' => $user->sesionesActivas()
                    ->where('session_id', '!=', $currentSessionId)
                    ->select('id', 'device_name', 'ip_address', 'last_activity', 'location')
                    ->get()
                    ->map(function ($session) {
                        return [
                            'id' => $session->id,
                            'device' => $session->device_name,
                            'ip' => $session->ip_address,
                            'location' => $session->location,
                            'last_activity' => $session->last_activity->diffForHumans(),
                        ];
                    })
            ];
        }

        // Limpiar los datos de la sesión
        $request->session()->forget('session_limit_data');

        return Inertia::render('Auth/SessionLimit', [
            'user_plan' => $user->plan,
            'sessions_limit' => $sessionLimitData['max_sessions'],
            'active_sessions_count' => $sessionLimitData['active_sessions'],
            'other_sessions' => $sessionLimitData['other_sessions'],
            'limit_exceeded' => true
        ]);
    }
}
