<?php

namespace App\Http\Controllers;

use App\Http\Traits\AlertResponseTrait;
use App\Http\Requests\ImportarExcelEnriquecimientoRequest;
use App\Http\Requests\ImportarManualEnriquecimientoRequest;
use App\Jobs\ProcesarEnriquecimientoNumerosJob;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Auth;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Facades\Log;
use App\Models\EnriquecimientoImportacion;
use App\Models\EnriquecimientoNumero;
use App\Models\UsuarioCredito;
use App\Exports\EnriquecimientoExport;
use Maatwebsite\Excel\Facades\Excel;
use Inertia\Inertia;
use Carbon\Carbon;

class EnriquecimientoController extends Controller
{
    use AlertResponseTrait;

    /**
     * Resumen del último procesamiento de Excel
     */
    private $resumenProcesamiento = null;

    /**
     * Normaliza la fecha específica de la petición
     */
    private function normalizeDateForDay($request)
    {
        $fechaDia = $request->input('fecha_dia') ?: Carbon::today()->toDateString();

        $fechaDiaCarbon = Carbon::parse($fechaDia);

        return $fechaDiaCarbon->toDateString();
    }

    // =======================
    // MÉTODOS PRINCIPALES DE VISTA
    // =======================

    /**
     * Display the enriquecimiento index with importaciones and statistics.
     */
    public function index(Request $request)
    {
        $user = Auth::user();

        // Si no hay parámetros de request, usar filtros de sesión
        if (!$request->hasAny(['fecha_dia', 'estado', 'tipo_importacion'])) {
            $fechaDia = session('enriquecimiento_fecha_dia', Carbon::today()->toDateString());
            $filtrosActivos = session('enriquecimiento_filtros', []);

            // Crear request con datos de sesión
            $request->merge([
                'fecha_dia' => $fechaDia,
                'estado' => isset($filtrosActivos['estado']) ? $filtrosActivos['estado'][0] ?? null : null,
                'tipo_importacion' => isset($filtrosActivos['tipo_importacion']) ? $filtrosActivos['tipo_importacion'][0] ?? null : null,
            ]);
        } else {
            // Normalizar fecha específica - por defecto hoy
            $fechaDia = $this->normalizeDateForDay($request);
            $filtrosActivos = [];
        }

        // Obtener estadísticas del usuario para la fecha específica
        $estadisticas = $this->getEstadisticasUsuario($user, $fechaDia);

        // Obtener importaciones con filtros por día específico
        $importaciones = $this->getImportacionesWithRelations($user, $fechaDia, $fechaDia, $request);

        // Obtener filtros activos de la sesión
        $filtrosActivos = session('enriquecimiento_filtros', []);

        return Inertia::render('GestionarEnriquecimiento', [
            'estadisticas' => $estadisticas,
            'importaciones' => $importaciones,
            'fechaDia' => $fechaDia,
            'filtrosActivos' => $filtrosActivos,
        ]);
    }

    /**
     * Procesar filtros y redirigir con datos filtrados
     */
    public function filtrar(Request $request)
    {
        $user = Auth::user();

        // Obtener filtros y fecha
        $filtros = $request->input('filtros', []);
        $fechaDia = $request->input('fecha_dia', Carbon::today()->toDateString());

        // Si los filtros están vacíos, limpiar la sesión
        if (
            empty($filtros) || (isset($filtros['estado']) && empty($filtros['estado'])) &&
            (isset($filtros['tipo_importacion']) && empty($filtros['tipo_importacion']))
        ) {
            session()->forget(['enriquecimiento_filtros', 'enriquecimiento_fecha_dia']);
            $filtrosActivos = [];
        } else {
            // Guardar filtros en sesión
            session([
                'enriquecimiento_filtros' => $filtros,
                'enriquecimiento_fecha_dia' => $fechaDia
            ]);
            $filtrosActivos = $filtros;
        }

        // Crear un request simulado con los filtros para reutilizar la lógica existente
        $filteredRequest = new Request();
        $filteredRequest->merge([
            'fecha_dia' => $fechaDia,
            'estado' => isset($filtros['estado']) ? $filtros['estado'][0] ?? null : null,
            'tipo_importacion' => isset($filtros['tipo_importacion']) ? $filtros['tipo_importacion'][0] ?? null : null,
        ]);

        // Obtener estadísticas del usuario para la fecha específica
        $estadisticas = $this->getEstadisticasUsuario($user, $fechaDia);

        // Obtener importaciones filtradas
        $importaciones = $this->getImportacionesWithRelations($user, $fechaDia, $fechaDia, $filteredRequest);

        return Inertia::render('GestionarEnriquecimiento', [
            'estadisticas' => $estadisticas,
            'importaciones' => $importaciones,
            'fechaDia' => $fechaDia,
            'filtrosActivos' => $filtrosActivos,
        ]);
    }

    /**
     * Mostrar detalle de una importación
     */
    public function detalles($id)
    {
        $user = Auth::user();

        $importacion = $this->getImportacionWithData($user, $id);
        $numeros = $this->getNumerosFromImportacion($importacion->id);

        return Inertia::render('DetallesEnriquecimiento', [
            'importacion' => $importacion,
            'numeros' => $numeros
        ]);
    }

    // =======================
    // MÉTODOS CRUD
    // =======================

    /**
     * Procesar importación de Excel
     */
    public function importarExcel(ImportarExcelEnriquecimientoRequest $request)
    {
        try {
            $resultado = $this->procesarImportacionExcel($request->validated());

            // Crear alerta según los créditos con el nuevo formato
            $alertData = [];
            if ($resultado['creditos_disponibles'] >= $resultado['contador']) {
                $alertData = [
                    'type' => 'success',
                    'title' => $resultado['titulo'],
                    'message' => $resultado['mensaje'],
                    'html' => true, // Permitir HTML en el mensaje
                    'timer' => 4000
                ];
            } elseif ($resultado['creditos_disponibles'] > 0) {
                $alertData = [
                    'type' => 'warning',
                    'title' => $resultado['titulo'],
                    'message' => $resultado['mensaje'],
                    'html' => true, // Permitir HTML en el mensaje
                    'timer' => 6000
                ];
            } else {
                $alertData = [
                    'type' => 'info',
                    'title' => $resultado['titulo'],
                    'message' => $resultado['mensaje'],
                    'html' => true, // Permitir HTML en el mensaje
                    'timer' => 5000
                ];
            }

            // Obtener datos actualizados para la vista
            $user = Auth::user();
            $fechaDia = Carbon::today()->toDateString();
            $estadisticas = $this->getEstadisticasUsuario($user, $fechaDia);
            $importaciones = $this->getImportacionesWithRelations($user, $fechaDia, $fechaDia);
            $filtrosActivos = session('enriquecimiento_filtros', []);

            return Inertia::render('GestionarEnriquecimiento', [
                'estadisticas' => $estadisticas,
                'importaciones' => $importaciones,
                'fechaDia' => $fechaDia,
                'filtrosActivos' => $filtrosActivos,
                'alert' => $alertData
            ]);
        } catch (\Exception $e) {
            return $this->redirectWithError('Error al procesar archivo', $e);
        }
    }

    /**
     * Procesar importación manual (textarea)
     */
    public function importarManual(ImportarManualEnriquecimientoRequest $request)
    {
        try {
            $resultado = $this->procesarImportacionManual($request->validated());

            // Crear alerta según los créditos con el nuevo formato
            $alertData = [];
            if ($resultado['creditos_disponibles'] >= $resultado['cantidad']) {
                $alertData = [
                    'type' => 'success',
                    'title' => $resultado['titulo'],
                    'message' => $resultado['mensaje'],
                    'html' => true, // Permitir HTML en el mensaje
                    'timer' => 4000
                ];
            } elseif ($resultado['creditos_disponibles'] > 0) {
                $alertData = [
                    'type' => 'warning',
                    'title' => $resultado['titulo'],
                    'message' => $resultado['mensaje'],
                    'html' => true, // Permitir HTML en el mensaje
                    'timer' => 6000
                ];
            } else {
                $alertData = [
                    'type' => 'info',
                    'title' => $resultado['titulo'],
                    'message' => $resultado['mensaje'],
                    'html' => true, // Permitir HTML en el mensaje
                    'timer' => 5000
                ];
            }

            // Obtener datos actualizados para la vista
            $user = Auth::user();
            $fechaDia = Carbon::today()->toDateString();
            $estadisticas = $this->getEstadisticasUsuario($user, $fechaDia);
            $importaciones = $this->getImportacionesWithRelations($user, $fechaDia, $fechaDia);
            $filtrosActivos = session('enriquecimiento_filtros', []);

            return Inertia::render('GestionarEnriquecimiento', [
                'estadisticas' => $estadisticas,
                'importaciones' => $importaciones,
                'fechaDia' => $fechaDia,
                'filtrosActivos' => $filtrosActivos,
                'alert' => $alertData
            ]);
        } catch (\Exception $e) {
            return $this->redirectWithError('Error al procesar números', $e);
        }
    }

    /**
     * Exportar resultados de una importación
     */
    public function exportar($id)
    {
        try {
            $user = Auth::user();

            // Verificar que la importación existe y pertenece al usuario
            $importacion = EnriquecimientoImportacion::where('user_id', $user->id)
                ->with(['numeros'])
                ->findOrFail($id);

            if ($importacion->numeros->isEmpty()) {
                return $this->redirectWithWarning('No hay números para exportar en esta importación');
            }

            $nombreArchivo = 'enriquecimiento_' . $importacion->id . '_' . now()->format('Y-m-d_H-i-s') . '.xlsx';

            return Excel::download(new \App\Exports\EnriquecimientoExport($id), $nombreArchivo);
        } catch (\Exception $e) {
            return $this->redirectWithError('Error al exportar', $e);
        }
    }

    /**
     * Eliminar una importación
     */
    public function destroy($id)
    {
        try {
            $user = Auth::user();

            // Validar que se puede eliminar
            $this->validateDeletion($user, $id);

            $this->eliminarImportacion($user, $id);

            return $this->redirectWithSuccess('Importación eliminada exitosamente');
        } catch (\Exception $e) {
            return $this->redirectWithError('Error al eliminar importación', $e);
        }
    }

    /**
     * Obtener datos actualizados para polling (AJAX)
     */
    public function obtenerDatosActualizados(Request $request)
    {
        try {
            $user = Auth::user();

            // Normalizar fecha específica
            $fechaDia = $this->normalizeDateForDay($request);

            // Obtener estadísticas actualizadas para la fecha específica
            $estadisticas = $this->getEstadisticasUsuario($user, $fechaDia);

            // Obtener importaciones actualizadas con filtros por día específico
            $importaciones = $this->getImportacionesWithRelations($user, $fechaDia, $fechaDia, $request);

            return response()->json([
                'estadisticas' => $estadisticas,
                'importaciones' => $importaciones,
                'timestamp' => now()->toISOString()
            ]);
        } catch (\Exception $e) {
            return response()->json([
                'error' => 'Error al obtener datos actualizados',
                'message' => $e->getMessage()
            ], 500);
        }
    }

    /**
     * Obtener datos actualizados de una importación específica para polling
     */
    public function obtenerDetallesActualizados(EnriquecimientoImportacion $importacion)
    {
        try {
            // Verificar que el usuario pueda acceder a esta importación
            if ($importacion->user_id !== Auth::id()) {
                return response()->json([
                    'error' => 'No autorizado'
                ], 403);
            }

            // Recargar la importación con sus relaciones
            $importacion->load('numeros');

            // Obtener números actualizados
            $numeros = $importacion->numeros()->orderBy('created_at', 'desc')->get();

            return response()->json([
                'importacion' => $importacion,
                'numeros' => $numeros,
                'timestamp' => now()->toISOString()
            ]);
        } catch (\Exception $e) {
            return response()->json([
                'error' => 'Error al obtener detalles actualizados',
                'message' => $e->getMessage()
            ], 500);
        }
    }

    // =======================
    // MÉTODOS AUXILIARES PARA DATOS
    // =======================

    private function getEstadisticasUsuario($user, $fechaDia = null)
    {
        // Usar el método centralizado de UsuarioCredito
        $estadisticasCreditos = UsuarioCredito::getEstadisticasUsuario($user->id);

        // Si se proporciona una fecha específica, calcular estadísticas para ese día
        if ($fechaDia) {
            $desde = Carbon::parse($fechaDia)->startOfDay();
            $hasta = Carbon::parse($fechaDia)->endOfDay();

            // Importaciones del día específico
            $importacionesTotales = EnriquecimientoImportacion::where('user_id', $user->id)
                ->whereBetween('created_at', [$desde, $hasta])
                ->count();

            // Números procesados del día específico
            $numerosProcesados = $this->getNumerosProcesadosDelDia($user->id, $fechaDia);

            return [
                'importaciones_totales' => $importacionesTotales,
                'numeros_procesados' => $numerosProcesados,
                'creditos' => $estadisticasCreditos,
                'numeros_disponibles_procesar' => UsuarioCredito::numerosDisponiblesParaProcesar($user->id),
            ];
        }

        // Si no hay fecha específica, usar estadísticas generales
        return [
            'importaciones_totales' => EnriquecimientoImportacion::where('user_id', $user->id)->count(),
            'numeros_procesados' => $this->getNumerosProcesados($user->id),
            'creditos' => $estadisticasCreditos,
            'numeros_disponibles_procesar' => UsuarioCredito::numerosDisponiblesParaProcesar($user->id),
        ];
    }

    private function getImportacionesWithRelations($user, $fechaDesde = null, $fechaHasta = null, $request = null)
    {
        $query = EnriquecimientoImportacion::where('user_id', $user->id)
            ->with('numeros');

        // Aplicar filtros de fecha si se proporcionan
        if ($fechaDesde && $fechaHasta) {
            $desde = Carbon::parse($fechaDesde)->startOfDay();
            $hasta = Carbon::parse($fechaHasta)->endOfDay();
            $query->whereBetween('created_at', [$desde, $hasta]);
        }

        // Aplicar filtros adicionales si se proporcionan en el request
        if ($request) {
            // Filtro por estado
            if ($request->filled('estado')) {
                $query->where('estado', $request->input('estado'));
            }

            // Filtro por tipo de importación
            if ($request->filled('tipo_importacion')) {
                $query->where('tipo_importacion', $request->input('tipo_importacion'));
            }

            // Búsqueda en observaciones o nombre de archivo
            if ($request->filled('search')) {
                $search = $request->input('search');
                $query->where(function ($q) use ($search) {
                    $q->where('observaciones', 'like', "%$search%")
                        ->orWhere('nombre_archivo', 'like', "%$search%");
                });
            }
        }

        $importaciones = $query->orderBy('created_at', 'desc')
            ->limit(50) // Aumentar el límite para mostrar más importaciones
            ->get();

        // Transformar los datos para que coincidan con lo que espera la vista
        return $importaciones->map(function ($importacion) {
            // Agregar campos que la vista espera (mapear desde nombres de BD)
            $importacion->procesados = $importacion->numeros_procesados;
            $importacion->pendientes = $importacion->numeros_pendientes;

            // Agregar campo errores contando números con estado rechazado
            $importacion->errores = $importacion->numeros()->where('estado', 'rechazado')->count();

            return $importacion;
        });
    }

    private function getImportacionWithNumeros($user, $id)
    {
        return EnriquecimientoImportacion::where('user_id', $user->id)
            ->with(['numeros' => function ($query) {
                $query->orderBy('fecha_registro', 'desc');
            }])
            ->findOrFail($id);
    }

    private function getNumerosProcesados($userId)
    {
        return EnriquecimientoNumero::whereHas('importacion', function ($q) use ($userId) {
            $q->where('user_id', $userId);
        })->where('estado', 'procesado')->count();
    }

    private function getNumerosProcesadosDelDia($userId, $fechaDia)
    {
        $desde = Carbon::parse($fechaDia)->startOfDay();
        $hasta = Carbon::parse($fechaDia)->endOfDay();

        return EnriquecimientoNumero::whereHas('importacion', function ($q) use ($userId, $desde, $hasta) {
            $q->where('user_id', $userId)
                ->whereBetween('created_at', [$desde, $hasta]);
        })->count();
    }

    private function getImportacionWithData($user, $id)
    {
        return EnriquecimientoImportacion::where('user_id', $user->id)
            ->findOrFail($id);
    }

    private function getNumerosFromImportacion($importacionId)
    {
        return EnriquecimientoNumero::where('importacion_id', $importacionId)
            ->orderBy('created_at', 'desc')
            ->get();
    }

    // =======================
    // MÉTODOS AUXILIARES PARA CRUD
    // =======================

    private function procesarImportacionExcel($data)
    {
        DB::beginTransaction();

        try {
            $user = Auth::user();
            $archivo = $data['archivo'];

            // Extraer números del Excel
            $numeros = $this->extraerNumerosDeExcel($archivo);

            if (empty($numeros)) {
                throw new \InvalidArgumentException('No se encontraron números válidos en el archivo');
            }

            // Crear importación SIN verificar créditos (solo al procesar)
            $importacion = $this->crearImportacion($user->id, [
                'nombre_archivo' => $archivo->getClientOriginalName(),
                'tipo_importacion' => 'excel',
                'total_numeros' => count($numeros)
            ]);

            // Guardar números como pendientes
            $this->guardarNumeros($importacion->id, $numeros);

            // Verificar créditos disponibles para mostrar advertencia
            $creditosDisponibles = UsuarioCredito::numerosDisponiblesParaProcesar($user->id);

            // Construir mensaje base con resumen del procesamiento
            $resumenMensaje = $this->construirMensajeResumen($archivo->getClientOriginalName(), count($numeros));

            if ($creditosDisponibles >= count($numeros)) {
                // Suficientes créditos - iniciar procesamiento automáticamente con delay
                ProcesarEnriquecimientoNumerosJob::dispatch($importacion)->delay(now()->addSeconds(2));
                $mensaje = $resumenMensaje['mensaje'] . "
                <div style='background: #f0fdf4; border: 1px solid #bbf7d0; border-radius: 6px; padding: 12px; margin-top: 15px;'>
                    <div style='color: #15803d; font-weight: 600; margin-bottom: 5px;'>Procesamiento iniciado automáticamente</div>
                    <div style='color: #16a34a; font-size: 13px;'>Los números comenzarán a procesarse en unos segundos</div>
                </div>";
            } elseif ($creditosDisponibles > 0) {
                // Créditos parciales - procesar lo que se puede con delay
                ProcesarEnriquecimientoNumerosJob::dispatch($importacion)->delay(now()->addSeconds(2));
                $mensaje = $resumenMensaje['mensaje'] . "
                <div style='background: #fffbeb; border: 1px solid #fed7aa; border-radius: 6px; padding: 12px; margin-top: 15px;'>
                    <div style='color: #d97706; font-weight: 600; margin-bottom: 5px;'>Procesamiento parcial iniciado</div>
                    <div style='color: #ea580c; font-size: 13px;'>{$creditosDisponibles} números se procesarán, " .
                    (count($numeros) - $creditosDisponibles) . " quedarán pendientes por créditos</div>
                </div>";
            } else {
                // Sin créditos - todo queda pendiente
                $mensaje = $resumenMensaje['mensaje'] . "
                <div style='background: #eff6ff; border: 1px solid #bfdbfe; border-radius: 6px; padding: 12px; margin-top: 15px;'>
                    <div style='color: #1d4ed8; font-weight: 600; margin-bottom: 5px;'>Números importados en espera</div>
                    <div style='color: #2563eb; font-size: 13px;'>Se procesarán cuando tengas créditos disponibles</div>
                </div>";
            }

            DB::commit();

            return [
                'contador' => count($numeros),
                'importacion_id' => $importacion->id,
                'mensaje' => $mensaje,
                'titulo' => $resumenMensaje['titulo'],
                'creditos_disponibles' => $creditosDisponibles
            ];
        } catch (\Exception $e) {
            DB::rollback();
            throw $e;
        }
    }

    private function procesarImportacionManual($data)
    {
        DB::beginTransaction();

        try {
            $user = Auth::user();

            // Extraer números del texto
            $resultadoExtraccion = $this->extraerNumerosDeTexto($data['numeros']);
            $numeros = $resultadoExtraccion['numerosValidos'];

            if (empty($numeros)) {
                throw new \InvalidArgumentException('No se encontraron números válidos en el texto proporcionado');
            }

            // Crear importación
            $importacion = $this->crearImportacion($user->id, [
                'nombre_archivo' => null,
                'tipo_importacion' => 'manual',
                'total_numeros' => count($numeros)
            ]);

            // Guardar números como pendientes
            $this->guardarNumeros($importacion->id, $numeros);

            // Verificar créditos disponibles
            $creditosDisponibles = UsuarioCredito::numerosDisponiblesParaProcesar($user->id);

            // Construir mensaje con resumen de procesamiento
            $resumenMensaje = $this->construirMensajeResumen('texto manual', count($numeros));

            if ($creditosDisponibles >= count($numeros)) {
                // Suficientes créditos - iniciar procesamiento automáticamente con delay
                ProcesarEnriquecimientoNumerosJob::dispatch($importacion)->delay(now()->addSeconds(2));
                $mensaje = $resumenMensaje['mensaje'] . "
                <div style='background: #f0fdf4; border: 1px solid #bbf7d0; border-radius: 6px; padding: 12px; margin-top: 15px;'>
                    <div style='color: #15803d; font-weight: 600; margin-bottom: 5px;'>Procesamiento iniciado automáticamente</div>
                    <div style='color: #16a34a; font-size: 13px;'>Los números comenzarán a procesarse en unos segundos</div>
                </div>";
            } elseif ($creditosDisponibles > 0) {
                // Créditos parciales con delay
                ProcesarEnriquecimientoNumerosJob::dispatch($importacion)->delay(now()->addSeconds(2));
                $mensaje = $resumenMensaje['mensaje'] . "
                <div style='background: #fffbeb; border: 1px solid #fed7aa; border-radius: 6px; padding: 12px; margin-top: 15px;'>
                    <div style='color: #d97706; font-weight: 600; margin-bottom: 5px;'>Procesamiento parcial iniciado</div>
                    <div style='color: #ea580c; font-size: 13px;'>{$creditosDisponibles} números se procesarán, " .
                    (count($numeros) - $creditosDisponibles) . " quedarán pendientes por créditos</div>
                </div>";
            } else {
                // Sin créditos
                $mensaje = $resumenMensaje['mensaje'] . "
                <div style='background: #eff6ff; border: 1px solid #bfdbfe; border-radius: 6px; padding: 12px; margin-top: 15px;'>
                    <div style='color: #1d4ed8; font-weight: 600; margin-bottom: 5px;'>Números importados en espera</div>
                    <div style='color: #2563eb; font-size: 13px;'>Se procesarán cuando tengas créditos disponibles</div>
                </div>";
            }

            DB::commit();

            return [
                'cantidad' => count($numeros),
                'importacion_id' => $importacion->id,
                'mensaje' => $mensaje,
                'titulo' => $resumenMensaje['titulo'],
                'creditos_disponibles' => $creditosDisponibles
            ];
        } catch (\Exception $e) {
            DB::rollback();
            throw $e;
        }
    }



    private function crearImportacion($userId, $data)
    {
        return EnriquecimientoImportacion::create([
            'user_id' => $userId,
            'nombre_archivo' => $data['nombre_archivo'],
            'tipo_importacion' => $data['tipo_importacion'],
            'total_numeros' => $data['total_numeros'],
            'numeros_pendientes' => $data['total_numeros'],
            'estado' => 'pendiente'
        ]);
    }

    private function guardarNumeros($importacionId, $numeros)
    {
        foreach ($numeros as $numero) {
            EnriquecimientoNumero::create([
                'importacion_id' => $importacionId,
                'numero_telefono' => $numero,
                'estado' => 'pendiente',
                'fecha_registro' => now()
            ]);
        }
    }

    private function eliminarImportacion($user, $id)
    {
        $importacion = EnriquecimientoImportacion::where('user_id', $user->id)->findOrFail($id);
        $importacion->delete();
    }

    // =======================
    // VALIDACIONES DE NEGOCIO
    // =======================

    private function validateDeletion($user, $id)
    {
        $importacion = EnriquecimientoImportacion::where('user_id', $user->id)->findOrFail($id);

        if (!in_array($importacion->estado, ['rechazado', 'procesado', 'cancelado'])) {
            throw new \InvalidArgumentException('No se puede eliminar una importación en procesamiento');
        }
    }

    // =======================
    // MÉTODOS AUXILIARES DE PROCESAMIENTO
    // =======================

    /**
     * Validar número móvil de España
     * Solo acepta números de 9 dígitos que comiencen con 6 o 7
     */
    private function validarNumeroMovilEspana($numero)
    {
        // Eliminar espacios en blanco y caracteres no numéricos
        $numero = preg_replace('/[^0-9]/', '', trim($numero));

        // Validar que sea un número y tenga 9 dígitos, comenzando con 6 o 7
        if (preg_match('/^(6|7)[0-9]{8}$/', $numero)) {
            return true; // Número válido
        } else {
            return false; // Número no válido
        }
    }

    private function extraerNumerosDeExcel($archivo)
    {
        try {
            $data = Excel::toArray([], $archivo);
            $numeros = [];
            $numerosInvalidos = [];
            $totalFilas = 0;

            if (!empty($data) && !empty($data[0])) {
                $primerFila = true;

                foreach ($data[0] as $index => $fila) {
                    $totalFilas++;

                    // Omitir la primera fila si contiene texto como cabecera
                    if ($primerFila) {
                        $primerFila = false;
                        $primerCelda = trim($fila[0] ?? '');

                        // Si la primera celda contiene texto no numérico, es una cabecera
                        if (!empty($primerCelda) && !is_numeric(preg_replace('/[^0-9]/', '', $primerCelda))) {
                            continue; // Saltar la cabecera
                        }
                    }

                    if (!empty($fila[0])) {
                        $numero = trim($fila[0]);
                        $numeroLimpio = preg_replace('/[^0-9]/', '', $numero);

                        if ($this->validarNumeroMovilEspana($numeroLimpio)) {
                            $numeros[] = $numeroLimpio;
                        } else {
                            $numerosInvalidos[] = [
                                'fila' => $index + 1,
                                'numero_original' => $numero,
                                'numero_limpio' => $numeroLimpio
                            ];
                        }
                    }
                }
            }

            // Remover duplicados
            $numerosUnicos = array_unique($numeros);
            $duplicadosEncontrados = count($numeros) - count($numerosUnicos);

            // Si no hay números válidos, rechazar
            if (empty($numerosUnicos)) {
                $mensajeError = "No se encontraron números móviles españoles válidos en el archivo.\n";
                $mensajeError .= "Total de filas procesadas: {$totalFilas}\n";
                $mensajeError .= "Números inválidos encontrados: " . count($numerosInvalidos) . "\n\n";
                $mensajeError .= "Los números deben ser móviles españoles de 9 dígitos comenzando con 6 o 7.\n";
                $mensajeError .= "Ejemplos válidos: 612345678, 723456789";

                throw new \InvalidArgumentException($mensajeError);
            }

            // Almacenar resumen para uso posterior
            $this->resumenProcesamiento = [
                'total_filas' => $totalFilas,
                'numeros_validos' => count($numerosUnicos),
                'numeros_invalidos' => count($numerosInvalidos),
                'duplicados_removidos' => $duplicadosEncontrados,
                'detalles_invalidos' => array_slice($numerosInvalidos, 0, 15) // Guardar primeros 15 para mostrar
            ];

            return $numerosUnicos;
        } catch (\Exception $e) {
            // Si ya es nuestro error de validación, re-lanzarlo
            if ($e instanceof \InvalidArgumentException) {
                throw $e;
            }
            // Si hay error leyendo Excel, lanzar excepción
            throw new \InvalidArgumentException('Error al leer archivo Excel: ' . $e->getMessage());
        }
    }

    private function extraerNumerosDeTexto($texto)
    {
        $lineas = explode("\n", $texto);
        $numeros = [];
        $numerosInvalidos = [];
        $totalLineas = 0;

        foreach ($lineas as $index => $linea) {
            $linea = trim($linea);
            if (!empty($linea)) {
                $totalLineas++;
                $numeroLimpio = preg_replace('/[^0-9]/', '', $linea);

                if ($this->validarNumeroMovilEspana($numeroLimpio)) {
                    $numeros[] = $numeroLimpio;
                } else {
                    $numerosInvalidos[] = [
                        'linea' => $index + 1,
                        'numero_original' => $linea,
                        'numero_limpio' => $numeroLimpio
                    ];
                }
            }
        }

        // Remover duplicados
        $numerosUnicos = array_unique($numeros);
        $duplicadosEncontrados = count($numeros) - count($numerosUnicos);

        // Si no hay números válidos, rechazar
        if (empty($numerosUnicos)) {
            $mensajeError = "No se encontraron números móviles españoles válidos.\n";
            $mensajeError .= "Total de líneas procesadas: {$totalLineas}\n";
            $mensajeError .= "Números inválidos encontrados: " . count($numerosInvalidos) . "\n\n";
            $mensajeError .= "Los números deben ser móviles españoles de 9 dígitos comenzando con 6 o 7.\n";
            $mensajeError .= "Ejemplos válidos: 612345678, 723456789";

            throw new \InvalidArgumentException($mensajeError);
        }

        // Almacenar resumen para uso posterior
        $this->resumenProcesamiento = [
            'total_lineas' => $totalLineas,
            'numeros_validos' => count($numerosUnicos),
            'numeros_invalidos' => count($numerosInvalidos),
            'duplicados_removidos' => $duplicadosEncontrados,
            'detalles_invalidos' => array_slice($numerosInvalidos, 0, 15) // Guardar primeros 15 para mostrar
        ];

        return $numerosUnicos;
    }

    private function normalizarNumero($numero)
    {
        $numero = preg_replace('/[^0-9+]/', '', $numero);

        if (strlen($numero) == 9 && !str_starts_with($numero, '+')) {
            $numero = '+51' . $numero;
        }

        return $numero;
    }

    // =======================
    // MÉTODOS DE PLANTILLA
    // =======================

    /**
     * Descargar plantilla Excel para importar números
     */
    public function descargarPlantilla()
    {
        try {
            // Crear un nuevo archivo Excel
            $spreadsheet = new \PhpOffice\PhpSpreadsheet\Spreadsheet();
            $sheet = $spreadsheet->getActiveSheet();

            // Establecer el título de la hoja
            $sheet->setTitle('Plantilla Enriquecimiento');

            // Cabeceras del archivo (solo número de teléfono)
            $headers = ['numero_telefono'];

            // Escribir cabeceras en la primera fila
            foreach ($headers as $index => $header) {
                $sheet->setCellValueByColumnAndRow($index + 1, 1, ucfirst(str_replace('_', ' ', $header)));
            }

            // Aplicar formato a las cabeceras
            $lastColumn = \PhpOffice\PhpSpreadsheet\Cell\Coordinate::stringFromColumnIndex(count($headers));
            $sheet->getStyle('A1:' . $lastColumn . '1')->applyFromArray([
                'font' => [
                    'bold' => true,
                    'color' => ['rgb' => 'FFFFFF']
                ],
                'fill' => [
                    'fillType' => \PhpOffice\PhpSpreadsheet\Style\Fill::FILL_SOLID,
                    'startColor' => ['rgb' => '4F46E5']
                ]
            ]);



            // Ajustar el ancho de las columnas automáticamente
            foreach (range(1, count($headers)) as $colIndex) {
                $sheet->getColumnDimensionByColumn($colIndex)->setAutoSize(true);
            }

            // Crear el writer para Excel
            $writer = new \PhpOffice\PhpSpreadsheet\Writer\Xlsx($spreadsheet);

            // Crear el nombre del archivo
            $filename = 'plantilla_enriquecimiento_' . date('Y-m-d') . '.xlsx';

            // Crear respuesta con el archivo Excel
            return response()->streamDownload(function () use ($writer) {
                $writer->save('php://output');
            }, $filename, [
                'Content-Type' => 'application/vnd.openxmlformats-officedocument.spreadsheetml.sheet',
                'Cache-Control' => 'max-age=0',
                'Expires' => '0',
            ]);
        } catch (\Exception $e) {
            return redirect()->back()->with('error', 'Error al generar la plantilla Excel: ' . $e->getMessage());
        }
    }

    /**
     * Construir mensaje de resumen del procesamiento
     */
    private function construirMensajeResumen($nombreArchivo, $numerosValidos)
    {
        if (!$this->resumenProcesamiento) {
            return [
                'titulo' => 'Archivo procesado exitosamente',
                'mensaje' => "<div style='text-align: center; padding: 10px;'>
                    <div style='background: #f0f9ff; border: 1px solid #bae6fd; border-radius: 8px; padding: 15px; margin: 10px 0;'>
                        <h4 style='color: #0369a1; margin: 0 0 8px 0; font-weight: 600;'>{$nombreArchivo}</h4>
                        <p style='color: #0f766e; margin: 0; font-size: 16px; font-weight: 500;'>
                            <span style='color: #16a34a;'>{$numerosValidos} números procesados</span>
                        </p>
                    </div>
                </div>"
            ];
        }

        $resumen = $this->resumenProcesamiento;

        // Título del mensaje
        $titulo = "Resumen de importación";

        // Construir mensaje HTML simple y elegante
        $mensaje = "<div style='text-align: left; font-family: system-ui, -apple-system, sans-serif;'>";

        // Cabecera del archivo
        $mensaje .= "<div style='background: #f8fafc; border: 1px solid #e2e8f0; border-radius: 8px; padding: 15px; margin-bottom: 15px;'>";
        $mensaje .= "<h4 style='color: #1e293b; margin: 0 0 5px 0; font-size: 16px; font-weight: 600;'>{$nombreArchivo}</h4>";
        $mensaje .= "<p style='color: #64748b; margin: 0; font-size: 13px;'>Archivo procesado correctamente</p>";
        $mensaje .= "</div>";

        // Estadísticas en formato simple
        $mensaje .= "<div style='display: flex; gap: 10px; flex-wrap: wrap; margin-bottom: 15px;'>";

        // Números válidos - siempre mostrar
        $mensaje .= "<div style='background: #f0fdf4; border: 1px solid #bbf7d0; border-radius: 6px; padding: 12px; flex: 1; min-width: 120px;'>";
        $mensaje .= "<div style='color: #15803d; font-weight: 600; font-size: 14px;'>{$numerosValidos}</div>";
        $mensaje .= "<div style='color: #16a34a; font-size: 12px;'>Números válidos</div>";
        $mensaje .= "</div>";

        // Duplicados si los hay
        if ($resumen['duplicados_removidos'] > 0) {
            $mensaje .= "<div style='background: #eff6ff; border: 1px solid #bfdbfe; border-radius: 6px; padding: 12px; flex: 1; min-width: 120px;'>";
            $mensaje .= "<div style='color: #1d4ed8; font-weight: 600; font-size: 14px;'>{$resumen['duplicados_removidos']}</div>";
            $mensaje .= "<div style='color: #2563eb; font-size: 12px;'>Duplicados removidos</div>";
            $mensaje .= "</div>";
        }

        $mensaje .= "</div>"; // Fin del flex container

        // Números inválidos si los hay
        if ($resumen['numeros_invalidos'] > 0) {
            $mensaje .= "<div style='background: #fffbeb; border: 1px solid #fed7aa; border-radius: 6px; padding: 12px; margin-bottom: 10px;'>";
            $mensaje .= "<div style='color: #d97706; font-weight: 600; margin-bottom: 5px;'>{$resumen['numeros_invalidos']} números omitidos</div>";
            $mensaje .= "<div style='color: #ea580c; font-size: 13px; margin-bottom: 8px;'>No son móviles españoles válidos</div>";

            // Mostrar algunos ejemplos
            if (!empty($resumen['detalles_invalidos'])) {
                $ejemplos = array_slice($resumen['detalles_invalidos'], 0, 2);
                $mensaje .= "<div style='background: #fef3c7; border-radius: 4px; padding: 8px; font-size: 12px;'>";
                $mensaje .= "<div style='color: #92400e; font-weight: 500; margin-bottom: 3px;'>Ejemplos omitidos:</div>";

                foreach ($ejemplos as $detalle) {
                    $mensaje .= "<div style='color: #b45309;'>• Fila {$detalle['fila']}: <code style='background: #fed7aa; padding: 1px 4px; border-radius: 2px;'>{$detalle['numero_original']}</code></div>";
                }

                if ($resumen['numeros_invalidos'] > 2) {
                    $mensaje .= "<div style='color: #a16207; font-style: italic; margin-top: 3px;'>... y " . ($resumen['numeros_invalidos'] - 2) . " más</div>";
                }
                $mensaje .= "</div>";
            }
            $mensaje .= "</div>";
        }

        $mensaje .= "</div>"; // Fin del contenedor principal

        return [
            'titulo' => $titulo,
            'mensaje' => $mensaje
        ];
    }

    /**
     * Pausar el procesamiento de una importación (cancelar definitivamente)
     */
    public function pausar($id)
    {
        try {
            $user = Auth::user();

            // Buscar la importación que pertenece al usuario
            $importacion = EnriquecimientoImportacion::where('id', $id)
                ->where('user_id', $user->id)
                ->first();

            if (!$importacion) {
                return redirect()->back()->with('alert', [
                    'type' => 'error',
                    'message' => 'Importación no encontrada',
                    'timer' => 4000
                ]);
            }

            // Verificar que esté en estado procesando
            if ($importacion->estado !== 'procesando') {
                return redirect()->back()->with('alert', [
                    'type' => 'error',
                    'message' => 'Solo se pueden cancelar importaciones en procesamiento',
                    'timer' => 4000
                ]);
            }

            DB::transaction(function () use ($importacion) {
                // Cambiar el estado de la importación a CANCELADO (definitivo)
                $importacion->update([
                    'estado' => 'cancelado',
                    'observaciones' => 'Procesamiento cancelado por el usuario',
                    'fecha_completado' => now()
                ]);

                // Cambiar todos los números pendientes y procesando a cancelado
                EnriquecimientoNumero::where('importacion_id', $importacion->id)
                    ->whereIn('estado', ['pendiente', 'procesando'])
                    ->update([
                        'estado' => 'cancelado',
                        'error_detalle' => 'Cancelado por el usuario',
                        'updated_at' => now()
                    ]);

                // Actualizar contadores
                $importacion->refresh();
                $rechazados = EnriquecimientoNumero::where('importacion_id', $importacion->id)
                    ->where('estado', 'rechazado')
                    ->count();

                $procesados = EnriquecimientoNumero::where('importacion_id', $importacion->id)
                    ->where('estado', 'procesado')
                    ->count();

                $importacion->update([
                    'numeros_procesados' => $procesados,
                    'numeros_pendientes' => 0 // Ya no hay pendientes
                ]);
            });

            return $this->redirectWithSuccess('Procesamiento cancelado correctamente. La importación ha sido cancelada definitivamente.', 'enriquecimiento.index');
        } catch (\Exception $e) {
            return $this->redirectWithError('Error al cancelar el procesamiento', $e);
        }
    }

    // =======================
    // MÉTODOS DE FILTRADO
    // =======================

    /**
     * Normaliza el rango de fechas de la petición
     */
    private function normalizeDateRange($request)
    {
        $fechaDesde = $request->input('fecha_desde') ?: Carbon::yesterday()->toDateString();
        $fechaHasta = $request->input('fecha_hasta') ?: Carbon::today()->toDateString();

        $fechaDesdeCarbon = Carbon::parse($fechaDesde);
        $fechaHastaCarbon = Carbon::parse($fechaHasta);

        if ($fechaDesdeCarbon->gt($fechaHastaCarbon)) {
            return [$fechaHasta, $fechaDesde];
        }

        return [$fechaDesde, $fechaHasta];
    }

    // =======================
    // MÉTODOS PARA BOT API
    // =======================

    /**
     * Procesar números desde el bot externo
     * POST /api/enriquecimiento/procesar
     */
    public function procesarBot(Request $request)
    {
        try {
            // Validar datos de entrada
            $request->validate([
                'user_id' => 'required|integer|exists:users,id',
                'numeros' => 'required|array|min:1',
                'numeros.*' => 'required|string',
                'metadata' => 'sometimes|array'
            ]);

            $userId = $request->input('user_id');
            $numerosRaw = $request->input('numeros');
            $metadata = $request->input('metadata', []);

            // Procesar y validar números
            $numerosValidos = [];
            $numerosInvalidos = [];

            foreach ($numerosRaw as $numero) {
                $numeroLimpio = preg_replace('/[^0-9]/', '', trim($numero));

                if ($this->validarNumeroMovilEspana($numeroLimpio)) {
                    $numerosValidos[] = $numeroLimpio;
                } else {
                    $numerosInvalidos[] = $numero;
                }
            }

            if (empty($numerosValidos)) {
                return response()->json([
                    'success' => false,
                    'error' => 'No se encontraron números válidos',
                    'numeros_invalidos' => $numerosInvalidos
                ], 400);
            }

            // Verificar créditos disponibles
            $creditosDisponibles = UsuarioCredito::numerosDisponiblesParaProcesar($userId);

            if ($creditosDisponibles < count($numerosValidos)) {
                return response()->json([
                    'success' => false,
                    'error' => 'Créditos insuficientes',
                    'creditos_disponibles' => $creditosDisponibles,
                    'creditos_necesarios' => count($numerosValidos)
                ], 402);
            }

            // Crear importación
            $importacion = $this->crearImportacion($userId, [
                'nombre_archivo' => 'bot_import_' . now()->format('Y-m-d_H-i-s'),
                'tipo_importacion' => 'bot',
                'total_numeros' => count($numerosValidos)
            ]);

            // Guardar números
            $this->guardarNumeros($importacion->id, $numerosValidos);

            // Actualizar metadata si se proporciona
            if (!empty($metadata)) {
                $importacion->update([
                    'observaciones' => json_encode($metadata)
                ]);
            }

            // Iniciar procesamiento inmediatamente
            ProcesarEnriquecimientoNumerosJob::dispatch($importacion);

            return response()->json([
                'success' => true,
                'importacion_id' => $importacion->id,
                'numeros_procesados' => count($numerosValidos),
                'creditos_utilizados' => count($numerosValidos),
                'creditos_restantes' => $creditosDisponibles - count($numerosValidos),
                'numeros_invalidos' => $numerosInvalidos
            ]);
        } catch (\Illuminate\Validation\ValidationException $e) {
            return response()->json([
                'success' => false,
                'error' => 'Datos de entrada inválidos',
                'validation_errors' => $e->errors()
            ], 422);
        } catch (\Exception $e) {
            return response()->json([
                'success' => false,
                'error' => 'Error interno del servidor',
                'message' => $e->getMessage()
            ], 500);
        }
    }

    /**
     * Actualizar estado de un número desde el bot
     * POST /api/enriquecimiento/actualizar-estado
     */
    public function actualizarEstadoBot(Request $request)
    {
        try {
            // Validar datos de entrada
            $request->validate([
                'numero_id' => 'required|integer|exists:enriquecimiento_numeros,id',
                'estado' => 'required|in:procesado,rechazado,cancelado',
                'datos_enriquecidos' => 'sometimes|array',
                'error_mensaje' => 'sometimes|string'
            ]);

            $numeroId = $request->input('numero_id');
            $estado = $request->input('estado');
            $datosEnriquecidos = $request->input('datos_enriquecidos', []);
            $errorMensaje = $request->input('error_mensaje');

            // Buscar el número
            $numero = EnriquecimientoNumero::findOrFail($numeroId);

            // Verificar que esté en procesamiento
            if ($numero->estado !== 'procesando') {
                return response()->json([
                    'success' => false,
                    'error' => 'El número no está en procesamiento',
                    'estado_actual' => $numero->estado
                ], 400);
            }

            // Actualizar el número
            $numero->update([
                'estado' => $estado,
                'datos_enriquecidos' => !empty($datosEnriquecidos) ? json_encode($datosEnriquecidos) : null,
                'error_mensaje' => $errorMensaje,
                'fecha_procesamiento' => now()
            ]);

            // Actualizar contadores de la importación
            $importacion = $numero->importacion;
            $this->actualizarContadoresImportacion($importacion);

            return response()->json([
                'success' => true,
                'numero_actualizado' => $numero->numero_telefono,
                'estado' => $estado,
                'importacion_id' => $importacion->id
            ]);
        } catch (\Illuminate\Validation\ValidationException $e) {
            return response()->json([
                'success' => false,
                'error' => 'Datos de entrada inválidos',
                'validation_errors' => $e->errors()
            ], 422);
        } catch (\Exception $e) {
            return response()->json([
                'success' => false,
                'error' => 'Error interno del servidor',
                'message' => $e->getMessage()
            ], 500);
        }
    }

    /**
     * Actualizar contadores de una importación
     */
    private function actualizarContadoresImportacion($importacion)
    {
        $numeros = $importacion->numeros;

        $numerosProcessados = $numeros->where('estado', 'procesado')->count();
        $numerosPendientes = $numeros->whereIn('estado', ['pendiente', 'procesando'])->count();
        $numerosRechazados = $numeros->where('estado', 'rechazado')->count();
        $numerosCancelados = $numeros->where('estado', 'cancelado')->count();

        // Determinar estado de la importación
        $estadoImportacion = 'pendiente';
        if ($numerosPendientes == 0) {
            if ($numerosProcessados > 0) {
                $estadoImportacion = 'procesado';
            } elseif ($numerosRechazados > 0 || $numerosCancelados > 0) {
                $estadoImportacion = 'rechazado';
            }
        } elseif ($numerosProcessados > 0 || $numerosRechazados > 0) {
            $estadoImportacion = 'procesando';
        }

        $importacion->update([
            'numeros_procesados' => $numerosProcessados,
            'numeros_pendientes' => $numerosPendientes,
            'estado' => $estadoImportacion,
            'fecha_completado' => $numerosPendientes == 0 ? now() : null
        ]);
    }

    /**
     * Reprocesar números rechazados por falta de créditos
     */
    public function reprocesarRechazados(EnriquecimientoImportacion $importacion)
    {
        try {
            $user = Auth::user();

            // Obtener números rechazados
            $numerosRechazados = EnriquecimientoNumero::where('importacion_id', $importacion->id)
                ->where('estado', 'rechazado')
                ->get();

            if ($numerosRechazados->isEmpty()) {
                return redirect()->back()->with('warning', 'No hay números rechazados para reprocesar.');
            }

            // Verificar créditos disponibles
            $creditosDisponibles = UsuarioCredito::numerosDisponiblesParaProcesar($user->id);

            if ($creditosDisponibles <= 0) {
                return redirect()->back()->with('error', 'No tienes créditos suficientes para reprocesar números.');
            }

            // Determinar cuántos números se pueden reprocesar
            $cantidadAReprocesar = min($numerosRechazados->count(), $creditosDisponibles);

            // Cambiar estado de los números seleccionados de rechazado a pendiente
            $numerosParaReprocesar = $numerosRechazados->take($cantidadAReprocesar);

            foreach ($numerosParaReprocesar as $numero) {
                $numero->update([
                    'estado' => 'pendiente',
                    'error_detalle' => null,
                    'fecha_procesamiento' => null
                ]);
            }

            // Actualizar estado de la importación si es necesario
            if ($importacion->estado === 'rechazado') {
                $importacion->update(['estado' => 'procesando']);
            }

            // Disparar job para procesar los números
            ProcesarEnriquecimientoNumerosJob::dispatch($importacion);

            // Crear mensaje de respuesta
            $mensaje = $this->construirMensajeReprocesamiento($cantidadAReprocesar, $numerosRechazados->count(), $creditosDisponibles);

            return redirect()->back()->with('success', $mensaje);
        } catch (\Exception $e) {
            Log::error('Error al reprocesar números rechazados', [
                'importacion_id' => $importacion->id,
                'error' => $e->getMessage()
            ]);

            return redirect()->back()->with('error', 'Error al reprocesar números: ' . $e->getMessage());
        }
    }

    /**
     * Reprocesar un número específico
     */
    public function reprocesarNumero(EnriquecimientoNumero $numero)
    {
        try {
            $user = Auth::user();

            // Verificar que el número esté en estado rechazado o error
            if (!in_array($numero->estado, ['rechazado', 'error', 'sin_datos'])) {
                return redirect()->back()->with('warning', 'Solo se pueden reprocesar números rechazados o con error.');
            }

            // Verificar créditos disponibles
            $creditosDisponibles = UsuarioCredito::numerosDisponiblesParaProcesar($user->id);

            if ($creditosDisponibles <= 0) {
                return redirect()->back()->with('error', 'No tienes créditos suficientes para reprocesar este número.');
            }

            // Cambiar estado a pendiente
            $numero->update([
                'estado' => 'pendiente',
                'error_detalle' => null,
                'fecha_procesamiento' => null
            ]);

            // Actualizar estado de la importación si es necesario
            $importacion = $numero->importacion;
            if ($importacion->estado === 'rechazado') {
                $importacion->update(['estado' => 'procesando']);
            }

            // Disparar job para procesar
            ProcesarEnriquecimientoNumerosJob::dispatch($importacion);

            return redirect()->back()->with('success', 'Número marcado para reprocesamiento.');
        } catch (\Exception $e) {
            Log::error('Error al reprocesar número', [
                'numero_id' => $numero->id,
                'error' => $e->getMessage()
            ]);

            return redirect()->back()->with('error', 'Error al reprocesar número: ' . $e->getMessage());
        }
    }

    /**
     * Reprocesar todos los números con error o pendientes
     */
    public function reprocesarTodos(EnriquecimientoImportacion $importacion)
    {
        try {
            $user = Auth::user();

            // Obtener números pendientes y con error
            $numerosParaReprocesar = EnriquecimientoNumero::where('importacion_id', $importacion->id)
                ->whereIn('estado', ['pendiente', 'error', 'sin_datos', 'rechazado'])
                ->get();

            if ($numerosParaReprocesar->isEmpty()) {
                return redirect()->back()->with('warning', 'No hay números para reprocesar.');
            }

            // Verificar créditos disponibles para números rechazados
            $numerosRechazados = $numerosParaReprocesar->where('estado', 'rechazado');
            $creditosDisponibles = UsuarioCredito::numerosDisponiblesParaProcesar($user->id);

            if ($numerosRechazados->count() > 0 && $creditosDisponibles <= 0) {
                return redirect()->back()->with('error', 'No tienes créditos suficientes para reprocesar números rechazados.');
            }

            // Limpiar datos de error para todos los números
            foreach ($numerosParaReprocesar as $numero) {
                $numero->update([
                    'estado' => 'pendiente',
                    'error_detalle' => null,
                    'fecha_procesamiento' => null
                ]);
            }

            // Actualizar estado de la importación
            $importacion->update(['estado' => 'procesando']);

            // Disparar job para procesar
            ProcesarEnriquecimientoNumerosJob::dispatch($importacion);

            return redirect()->back()->with('success', 'Todos los números han sido marcados para reprocesamiento.');
        } catch (\Exception $e) {
            Log::error('Error al reprocesar todos los números', [
                'importacion_id' => $importacion->id,
                'error' => $e->getMessage()
            ]);

            return redirect()->back()->with('error', 'Error al reprocesar números: ' . $e->getMessage());
        }
    }

    /**
     * Construir mensaje para reprocesamiento
     */
    private function construirMensajeReprocesamiento($cantidadReprocesada, $totalRechazados, $creditosDisponibles)
    {
        if ($cantidadReprocesada === $totalRechazados) {
            return "Se marcaron {$cantidadReprocesada} números rechazados para reprocesamiento.";
        } else {
            $restantes = $totalRechazados - $cantidadReprocesada;
            return "Se marcaron {$cantidadReprocesada} números para reprocesamiento. Quedan {$restantes} números rechazados sin reprocesar por falta de créditos.";
        }
    }
}
