<?php

namespace App\Http\Controllers;

use App\Http\Traits\AlertResponseTrait;
use App\Http\Requests\StorePlanRequest;
use App\Http\Requests\UpdatePlanRequest;
use App\Models\Plan;
use App\Models\User;
use Inertia\Inertia;

class PlanController extends Controller
{
    use AlertResponseTrait;
    /**
     * Muestra la lista de planes
     */
    public function index()
    {
        $planes = Plan::withCount('users')->orderBy('orden')->get();

        // Estadísticas para el dashboard
        $stats = [
            'total_planes' => Plan::count(),
            'planes_activos' => Plan::where('activo', true)->count(),
            'usuarios_con_plan' => User::whereNotNull('plan_id')->count(),
            'usuarios_sin_plan' => User::whereNull('plan_id')->count(),
        ];

        return Inertia::render('GestionarPlanes', [
            'planes' => $planes,
            'stats' => $stats
        ]);
    }

    /**
     * Crea un nuevo plan
     */
    public function store(StorePlanRequest $request)
    {
        try {
            $plan = Plan::create($request->validated());

            return $this->redirectWithSuccess('Plan creado correctamente', 'planes.index');
        } catch (\Exception $e) {
            return $this->redirectWithError('Error al crear plan', $e);
        }
    }

    /**
     * Actualiza un plan existente
     */
    public function update(UpdatePlanRequest $request, Plan $plan)
    {
        try {
            $plan->update($request->validated());

            return $this->redirectWithSuccess('Plan actualizado correctamente', 'planes.index');
        } catch (\Exception $e) {
            return $this->redirectWithError('Error al actualizar plan', $e);
        }
    }

    /**
     * Elimina un plan (soft delete)
     */
    public function destroy(Plan $plan)
    {
        try {
            // Verificar que no haya usuarios con este plan
            $usuariosConPlan = User::where('plan_id', $plan->id)->count();

            if ($usuariosConPlan > 0) {
                return redirect()
                    ->route('planes.index')
                    ->with('alert', [
                        'type' => 'warning',
                        'message' => "No se puede eliminar el plan porque hay {$usuariosConPlan} usuario(s) asignado(s) a este plan.",
                        'mode' => 'toast',
                        'timer' => 4000
                    ]);
            }

            $plan->delete();

            return $this->redirectWithSuccess('Plan eliminado correctamente', 'planes.index');
        } catch (\Exception $e) {
            return $this->redirectWithError('Error al eliminar plan', $e);
        }
    }
}
