<?php

namespace App\Http\Controllers;

use App\Models\User;
use App\Models\TipoDocumento;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Auth;
use Illuminate\Support\Facades\Gate;
use Illuminate\Support\Facades\Log;
use Inertia\Inertia;

class PublicProfileController extends Controller
{
    /**
     * Mostrar todos los perfiles públicos
     */
    public function index(Request $request)
    {
        $query = User::where('active', true)
            ->with(['roles', 'tipoDocumento'])
            ->select([
                'id',
                'name',
                'email',
                'tipo_documento_id',
                'numero_documento',
                'bio',
                'avatar',
                'phone',
                'position',
                'department',
                'is_public_profile',
                'created_at'
            ]);

        // Si no tiene permisos para ver todos, solo mostrar perfiles públicos
        if (!Gate::allows('profiles.ver-todos')) {
            $query->where('is_public_profile', true);
        }

        Log::info('Filtro profileType recibido', [
            'profileType' => $request->profileType,
            'permiso_ver_todos' => Gate::allows('profiles.ver-todos'),
        ]);

        $profiles = $query->when($request->search, function ($query, $search) {
            $query->where(function ($q) use ($search) {
                $q->where('name', 'like', "%{$search}%")
                    ->orWhere('email', 'like', "%{$search}%")
                    ->orWhere('numero_documento', 'like', "%{$search}%")
                    ->orWhere('position', 'like', "%{$search}%")
                    ->orWhere('department', 'like', "%{$search}%");
            });
        })
            ->when($request->department, function ($query, $department) {
                $query->where('department', $department);
            })
            ->when($request->position, function ($query, $position) {
                $query->where('position', $position);
            })
            ->when($request->profileType && Gate::allows('profiles.ver-todos'), function ($query) use ($request) {
                $profileType = $request->profileType;
                Log::info('Aplicando filtro profileType', [
                    'profileType' => $profileType
                ]);
                if ($profileType === 'public') {
                    $query->where('is_public_profile', true);
                } elseif ($profileType === 'private') {
                    $query->where('is_public_profile', false);
                }
                // Si es 'all' o cualquier otro valor, no aplicar filtro adicional
            })
            ->orderBy('name')
            ->paginate(12);

        // Para los filtros, aplicar las mismas restricciones
        $departmentQuery = User::where('active', true)->whereNotNull('department')->distinct();
        $positionQuery = User::where('active', true)->whereNotNull('position')->distinct();

        if (!Gate::allows('profiles.ver-todos')) {
            $departmentQuery->where('is_public_profile', true);
            $positionQuery->where('is_public_profile', true);
        }

        $departments = $departmentQuery->pluck('department');
        $positions = $positionQuery->pluck('position');

        return Inertia::render('PublicProfiles/Index', [
            'profiles' => $profiles,
            'departments' => $departments,
            'positions' => $positions,
            'filters' => $request->only(['search', 'department', 'position', 'profileType']),
            'permissions' => [
                'canViewDirectory' => Gate::allows('profiles.ver-directorio'),
                'canViewPublicProfile' => Gate::allows('profiles.ver-perfil-publico'),
                'canViewAll' => Gate::allows('profiles.ver-todos'),
            ]
        ]);
    }

    /**
     * Mostrar un perfil público específico
     */
    public function show(User $user)
    {
        // Si el usuario no está activo, no mostrar
        if (!$user->active) {
            abort(404, 'Perfil no encontrado');
        }

        // Si no tiene permisos para ver todos los perfiles, verificar que sea público
        if (!Gate::allows('profiles.ver-todos') && !$user->is_public_profile) {
            abort(404, 'Perfil no encontrado o no público');
        }

        $user->load(['roles', 'tipoDocumento']);

        $userData = $user->only([
            'id',
            'name',
            'email',
            'tipo_documento_id',
            'numero_documento',
            'bio',
            'avatar',
            'phone',
            'position',
            'department',
            'is_public_profile',
            'created_at',
            'email_verified_at'
        ]);

        $userData['roles'] = $user->roles;
        $userData['tipo_documento'] = $user->tipoDocumento;

        return Inertia::render('PublicProfiles/Show', [
            'user' => $userData,
            'permissions' => [
                'canViewDirectory' => Gate::allows('profiles.ver-directorio'),
                'canViewPublicProfile' => Gate::allows('profiles.ver-perfil-publico'),
                'canViewAll' => Gate::allows('profiles.ver-todos'),
            ]
        ]);
    }

    /**
     * Mostrar formulario de configuración de perfil público
     */
    public function editSettings()
    {
        $tiposDocumento = TipoDocumento::activos()->ordenado()->get();
        /** @var User $user */
        $user = Auth::user();
        $user->load('tipoDocumento');

        return Inertia::render('PublicProfiles/Settings', [
            'user' => $user,
            'tiposDocumento' => $tiposDocumento
        ]);
    }

    /**
     * Actualizar configuración de perfil público del usuario actual
     */
    public function updateSettings(Request $request)
    {
        $request->validate([
            'name' => 'required|string|max:255',
            'email' => 'required|email|unique:users,email,' . Auth::id(),
            'tipo_documento_id' => 'nullable|exists:tipos_documento,id',
            'numero_documento' => 'nullable|string|max:255|unique:users,numero_documento,' . Auth::id(),
            'is_public_profile' => 'boolean',
            'bio' => 'nullable|string|max:500',
            'phone' => 'nullable|string|max:20',
            'position' => 'nullable|string|max:100',
            'department' => 'nullable|string|max:100',
            'avatar' => 'nullable|image|mimes:jpeg,png,jpg,gif|max:2048'
        ]);

        /** @var User $user */
        $user = Auth::user();

        $data = $request->only([
            'name',
            'email',
            'tipo_documento_id',
            'numero_documento',
            'is_public_profile',
            'bio',
            'phone',
            'position',
            'department'
        ]);

        // Manejar subida de avatar
        if ($request->hasFile('avatar')) {
            // Eliminar avatar anterior si existe
            if ($user->avatar && file_exists(public_path('storage/' . $user->avatar))) {
                unlink(public_path('storage/' . $user->avatar));
            }

            $data['avatar'] = $request->file('avatar')->store('avatars', 'public');
        }

        $user->update($data);

        return back()->with('success', 'Configuración de perfil actualizada correctamente');
    }
}
