<?php

namespace App\Http\Controllers;

use App\Http\Traits\AlertResponseTrait;
use App\Http\Requests\StoreRoleRequest;
use App\Http\Requests\UpdateRoleRequest;
use App\Models\Role;
use App\Models\Cartera;
use App\Models\Reporte;
use App\Models\Permission;
use Inertia\Inertia;
use Inertia\Response;
use Illuminate\Support\Facades\Log;

class RoleController extends Controller
{
    use AlertResponseTrait;

    // =======================
    // MÉTODOS PRINCIPALES DE VISTA
    // =======================

    /**
     * Display the roles index with all roles, carteras, reportes y permisos.
     */
    public function index(): Response
    {
        // Obtener roles con relaciones
        $roles = $this->getRolesWithRelations();

        // Obtener datos auxiliares para formularios
        $formData = $this->getFormData();

        return Inertia::render('GestionarRoles', [
            'roles' => $roles,
            'carteras' => $formData['carteras'],
            'reportes' => $formData['reportes'],
            'permissions' => $formData['permissions'],
        ]);
    }

    // =======================
    // MÉTODOS CRUD
    // =======================

    /**
     * Store a newly created role in storage.
     */
    public function store(StoreRoleRequest $request)
    {
        try {
            // Crear rol y sincronizar relaciones
            $role = $this->createRole($request->validated());
            $this->syncRoleRelations($role, $request->validated());

            return $this->redirectWithSuccess("Rol creado correctamente", 'roles.index');
        } catch (\Exception $e) {
            Log::error('Error al crear rol: ' . $e->getMessage());
            return $this->redirectWithError('Error al crear el rol');
        }
    }

    /**
     * Update the specified role in storage.
     */
    public function update(UpdateRoleRequest $request, Role $role)
    {
        try {
            // Actualizar datos y sincronizar relaciones
            $this->updateRoleBasicData($role, $request->validated());
            $this->syncRoleRelations($role, $request->validated());

            return $this->redirectWithSuccess("Rol actualizado correctamente", 'roles.index');
        } catch (\Exception $e) {
            Log::error('Error al actualizar rol: ' . $e->getMessage());
            return $this->redirectWithError('Error al actualizar el rol');
        }
    }

    /**
     * Remove the specified role from storage.
     */
    public function destroy(Role $role)
    {
        try {
            $role->delete();
            return $this->redirectWithSuccess("Rol eliminado correctamente", 'roles.index');
        } catch (\Exception $e) {
            Log::error('Error al eliminar rol: ' . $e->getMessage());
            return $this->redirectWithError('Error al eliminar el rol');
        }
    }

    // =======================
    // MÉTODOS AUXILIARES PARA DATOS
    // =======================

    private function getRolesWithRelations()
    {
        return Role::with(['carteras', 'reportes', 'permissions'])->get();
    }

    private function getFormData()
    {
        return [
            'carteras' => Cartera::all(),
            'reportes' => Reporte::with('cartera')->get(),
            'permissions' => Permission::with('module')->get(),
        ];
    }

    // =======================
    // MÉTODOS AUXILIARES PARA CRUD
    // =======================

    private function createRole($data)
    {
        return Role::create([
            'name' => $data['name'],
            'guard_name' => 'web',
        ]);
    }

    private function updateRoleBasicData($role, $data)
    {
        $role->update([
            'name' => $data['name'],
            'guard_name' => 'web',
        ]);
    }

    // =======================
    // MÉTODOS AUXILIARES PARA RELACIONES
    // =======================

    private function syncRoleRelations($role, $data)
    {
        // Sincronizar relaciones pivot
        $role->carteras()->sync($data['carteras'] ?? []);
        $role->reportes()->sync($data['reportes'] ?? []);

        // Sincronizar permisos usando Spatie
        $role->syncPermissions($data['permissions'] ?? []);
    }
}
