<?php

namespace App\Http\Controllers;

use App\Models\Vodafone;
use App\Models\User;
use App\Models\TipoDocumento;
use App\Models\VodafoneAsignacion;
use App\Models\VodafoneAuditoria;
use Illuminate\Http\Request;
use Inertia\Inertia;
use Illuminate\Support\Facades\Auth;
use Illuminate\Support\Facades\Log;
use Carbon\Carbon;

class VodafoneController extends Controller
{
    // =======================
    // MÉTODOS PRINCIPALES DE VISTA
    // =======================

    public function index(Request $request)
    {
        /** @var \App\Models\User $user */
        $user = Auth::user();

        // Log de todos los datos recibidos
        Log::info('[VodafoneController] Request recibido', [
            'all_data' => $request->all(),
            'filtros_avanzados' => $request->input('filtros_avanzados'),
            'method' => $request->method(),
        ]);

        // Normalizar fechas
        [$fechaDesde, $fechaHasta] = $this->normalizeDateRange($request);

        // Construir query según filtros y permisos
        $query = $this->buildQuery($user, $fechaDesde, $fechaHasta, $request);

        // Obtener datos paginados
        $paginatedData = $this->getPaginatedData($query, $request);

        // Obtener usuarios asignables
        $usuariosAsignables = $this->getAssignableUsers();

        // Obtener tipos de documento
        $tiposDocumento = TipoDocumento::activos()->ordenado()->get();

        // Extraer configuración completa del request
        $configuracion = [
            'filtros' => [],
            'busqueda' => [
                'campo' => '',
                'valor' => '',
            ],
            'fechas' => [
                'desde' => $fechaDesde,
                'hasta' => $fechaHasta,
            ],
        ];

        // Filtros directos de trazabilidad
        if ($request->filled('trazabilidad')) {
            $configuracion['filtros']['trazabilidad'] = $request->input('trazabilidad');
        }

        // Filtros directos de upload_id
        if ($request->filled('upload_id')) {
            $configuracion['filtros']['upload_id'] = $request->input('upload_id');
        }

        // Búsqueda directa
        if ($request->filled('search_campo') && $request->filled('search_valor')) {
            $configuracion['busqueda']['campo'] = $request->input('search_campo');
            $configuracion['busqueda']['valor'] = $request->input('search_valor');
        }

        // Filtros avanzados (convertir a filtros directos o búsqueda)
        $filtrosAvanzados = $request->input('filtros_avanzados', []);
        if (is_array($filtrosAvanzados) && count($filtrosAvanzados)) {
            foreach ($filtrosAvanzados as $filtro) {
                if (isset($filtro['campo']) && isset($filtro['valor']) && $filtro['valor'] !== '') {
                    // Campos de búsqueda - van a busqueda
                    if (in_array($filtro['campo'], ['asignado_a_id', 'nombre_cliente', 'numero_documento', 'telefono_principal'])) {
                        $configuracion['busqueda']['campo'] = $filtro['campo'];
                        $configuracion['busqueda']['valor'] = $filtro['valor'];
                    }
                    // Filtros de estado - van a filtros
                    elseif ($filtro['campo'] === 'trazabilidad') {
                        if (!isset($configuracion['filtros']['trazabilidad'])) {
                            $configuracion['filtros']['trazabilidad'] = [];
                        }
                        if (!is_array($configuracion['filtros']['trazabilidad'])) {
                            $configuracion['filtros']['trazabilidad'] = [$configuracion['filtros']['trazabilidad']];
                        }
                        $configuracion['filtros']['trazabilidad'][] = $filtro['valor'];
                    }
                    // Otros filtros directos
                    elseif ($filtro['campo'] === 'upload_id') {
                        $configuracion['filtros']['upload_id'] = $filtro['valor'];
                    }
                }
            }
        }

        // --- AGREGACIÓN PARA GRÁFICO DE FICHAS COMPLETADAS POR ASESOR Y FECHA ---
        $usuarios = \App\Models\User::where('active', true)
            ->permission('vodafone.recibe-asignacion')
            ->with(['roles'])
            ->get(['id', 'name']);

        // Convertir fechas correctamente para incluir todo el día de "hasta"
        $fechaDesdeCarbon = Carbon::parse($fechaDesde)->startOfDay();
        $fechaHastaCarbon = Carbon::parse($fechaHasta)->endOfDay();

        $data = \App\Models\Vodafone::whereBetween('created_at', [$fechaDesdeCarbon, $fechaHastaCarbon])
            ->where('trazabilidad', 'completado')
            ->whereIn('asignado_a_id', $usuarios->pluck('id'))
            ->selectRaw('DATE(created_at) as fecha, asignado_a_id, count(*) as cantidad')
            ->groupBy('fecha', 'asignado_a_id')
            ->get();

        $fechas = collect();
        $usuariosNombres = $usuarios->pluck('name', 'id');
        $tabla = [];
        foreach ($data as $row) {
            $fechas->push($row->fecha);
        }
        $fechas = $fechas->unique()->sort()->values();
        foreach ($fechas as $fecha) {
            $fila = ['fecha' => $fecha];
            foreach ($usuarios as $u) {
                $fila[$u->name] = 0;
            }
            foreach ($data->where('fecha', $fecha) as $row) {
                $fila[$usuariosNombres[$row->asignado_a_id]] = $row->cantidad;
            }
            $tabla[] = $fila;
        }

        //data recibida
        Log::info('[VodafoneController] Datos para gráfico de fichas por asesor', [
            'data' => $data,
            //mostrar datos de filtro
            'configuracion' => $configuracion,
        ]);

        // Renderizar vista
        return Inertia::render('Vodafone', [
            'paginatedData' => $paginatedData,
            'success' => session('success'),
            'canViewGlobal' => $user->can('vodafone.ver-global'),
            'canAssign' => $user->can('vodafone.asignar'),
            'usuariosAsignables' => $usuariosAsignables,
            'tiposDocumento' => $tiposDocumento,
            'configuracion' => $configuracion, // ✅ CONFIGURACIÓN COMPLETA (incluye fechas)
            'asignado_a_id' => $configuracion['filtros']['asignado_a_id'] ?? null,
            // Fechas por compatibilidad temporal
            'fechaDesde' => $fechaDesde,
            'fechaHasta' => $fechaHasta,
            // Datos de importación
            'preview' => session('preview'),
            'truncado' => session('truncado'),
            'total' => session('total'),
            'total_registros' => session('total_registros'),
            'total_duplicados' => session('total_duplicados'),
            'total_nuevos' => session('total_nuevos'),
            'importData' => session('importData'),
            'log_id' => session('log_id'),
            // Datos para gráfico
            'graficoFichasPorAsesor' => [
                'usuarios' => $usuarios->pluck('name'),
                'fechas' => $fechas,
                'tabla' => $tabla,
            ],
        ]);
    }

    // =======================
    // MÉTODOS CRUD
    // =======================

    public function store(Request $request)
    {
        $data = $request->validate($this->validationRules());
        $data['user_id'] = Auth::id();

        Vodafone::create($data);

        return redirect()->back()
            ->with('success', 'Registro creado correctamente.');
    }

    public function update(Request $request, Vodafone $vodafone)
    {

        Log::info('[VodafoneController] Actualizando registro', [
            'vodafone_id' => $vodafone->id,
            'user_id' => Auth::id(),
            'data' => $request->all(),
        ]);
        $data = $request->validate($this->validationRules($vodafone->id));
        $user = Auth::user();
        // No sobrescribir el user_id (usuario creador) al actualizar
        unset($data['user_id']);
        // Remover asignado_a_id de los datos de actualización
        unset($data['asignado_a_id']);

        // Validar permisos de edición
        $validationResult = $this->validateEditPermissions($vodafone, $user, $data);
        if ($validationResult !== true) {
            return $validationResult; // Retorna el error
        }

        // Detectar cambios ANTES de auto-completar
        $cambios = $this->detectChanges($vodafone, $data);

        // Auto-completar si todos los campos están llenos
        $data = $this->autoCompleteIfReady($data);

        // Si autoCompleteIfReady cambió la trazabilidad, detectar este cambio también
        if (isset($data['trazabilidad']) && $data['trazabilidad'] !== $vodafone->trazabilidad) {
            if (!in_array('trazabilidad', $cambios['campos'])) {
                $cambios['campos'][] = 'trazabilidad';
                $cambios['cambios']['trazabilidad'] = [
                    'old' => $vodafone->trazabilidad,
                    'new' => $data['trazabilidad'],
                ];
            }
        }

        // Actualizar registro
        $vodafone->update($data);

        // Guardar auditoría si hay cambios
        if (!empty($cambios['campos'])) {
            $this->saveAuditTrail($vodafone, $user, $cambios['cambios']);
        }

        return redirect()->back()
            ->with('success', 'Registro actualizado correctamente.');
    }

    public function destroy(Request $request, Vodafone $vodafone)
    {
        /** @var \App\Models\User $user */
        $user = Auth::user();

        if ($vodafone->user_id !== $user->id && !$user->can('vodafone.eliminar')) {
            abort(403, 'No autorizado');
        }

        $vodafone->delete();

        return redirect()->back()
            ->with('success', 'Registro eliminado correctamente.');
    }

    // =======================
    // MÉTODOS DE ASIGNACIÓN
    // =======================

    public function asignar(Request $request)
    {
        $validatedData = $request->validate([
            'ids' => 'required|array|min:1',
            'ids.*' => 'integer|exists:historial_registros_vodafone,id',
            'asignado_a_id' => 'required|exists:users,id',
        ]);

        /** @var \App\Models\User $user */
        $user = Auth::user();
        $ids = $request->input('ids');
        $asignadoA = $request->input('asignado_a_id');

        $cantidad = $this->processAssignments($ids, $asignadoA, $user);

        $this->logAssignmentActivity($ids, $asignadoA, $user->id, $cantidad);

        return redirect()->back()
            ->with('success', "{$cantidad} registro(s) asignado(s) correctamente.");
    }

    public function agendar(Request $request, Vodafone $vodafone)
    {
        /** @var \App\Models\User $user */
        $user = Auth::user();

        // Verificar permisos
        if (!$user->can('vodafone.agendar')) {
            abort(403, 'No tienes permisos para agendar registros');
        }

        // Verificar que el registro esté completado
        if ($vodafone->trazabilidad !== 'completado') {
            return redirect()->back()
                ->with('error', 'Solo se pueden agendar registros completados.');
        }

        // Cambiar la trazabilidad a 'agendado'
        $trazabilidadAnterior = $vodafone->trazabilidad;
        $vodafone->update([
            'trazabilidad' => 'agendado',
            'updated_at' => now(),
        ]);

        // Registrar en auditoría
        $this->saveScheduleAuditTrail($vodafone, $user, $trazabilidadAnterior);

        return redirect()->back()
            ->with('success', 'Registro agendado correctamente.');
    }

    // =======================
    // MÉTODOS AUXILIARES UNIFICADOS
    // =======================

    /**
     * Normaliza el rango de fechas del request
     */
    private function normalizeDateRange($request)
    {
        $fechaDesde = $request->input('fecha_desde') ?: Carbon::yesterday()->toDateString();
        $fechaHasta = $request->input('fecha_hasta') ?: Carbon::today()->toDateString();

        $fechaDesdeCarbon = Carbon::parse($fechaDesde);
        $fechaHastaCarbon = Carbon::parse($fechaHasta);

        if ($fechaDesdeCarbon->gt($fechaHastaCarbon)) {
            return [$fechaHasta, $fechaDesde];
        }

        return [$fechaDesde, $fechaHasta];
    }

    /**
     * Construye la query principal con todos los filtros
     */
    private function buildQuery($user, $fechaDesde, $fechaHasta, $request)
    {
        $desde = Carbon::parse($fechaDesde)->startOfDay();
        $hasta = Carbon::parse($fechaHasta)->endOfDay();

        $query = Vodafone::query()->with(['asignado_a', 'user', 'tipoDocumento']);

        // Aplicar filtros de búsqueda
        $this->applySearchFilters($query, $request);

        // Aplicar filtros de trazabilidad
        $this->applyTraceabilityFilters($query, $request);

        // Aplicar filtros directos
        $this->applyDirectFilters($query, $request);

        // Filtros avanzados
        $this->applyAdvancedFilters($query, $request);

        // Aplicar filtros según permisos del usuario
        $this->applyPermissionFilters($query, $user, $desde, $hasta);

        return $query;
    }

    /**
     * Aplica filtros avanzados enviados como array de {campo, valor}
     */
    private function applyAdvancedFilters($query, $request)
    {
        $filtrosAvanzados = $request->input('filtros_avanzados', []);
        if (!is_array($filtrosAvanzados)) return;
        foreach ($filtrosAvanzados as $filtro) {
            if (!isset($filtro['campo']) || !isset($filtro['valor']) || $filtro['valor'] === '' || $filtro['valor'] === null) continue;
            $campo = $filtro['campo'];
            $valor = $filtro['valor'];
            // Forzar a número si corresponde
            if ($campo === 'asignado_a_id' && is_numeric($valor)) {
                $valor = (int)$valor;
            }
            // Campos directos
            if (in_array($campo, ['user_id', 'asignado_a_id', 'numero_documento', 'nombre_cliente', 'trazabilidad', 'upload_id'])) {
                $query->where($campo, $valor);
            }
            // Filtro por nombre de usuario creador
            else if ($campo === 'user_name') {
                $query->whereHas('user', function ($q) use ($valor) {
                    $q->where('name', 'like', "%$valor%");
                });
            }
            // Filtro por nombre de usuario asignado
            else if ($campo === 'asignado_a_name') {
                $query->whereHas('asignado_a', function ($q) use ($valor) {


                    $q->where('name', 'like', "%$valor%");
                });
            }
            // Agrega más casos según tus relaciones/campos
        }
    }

    /**
     * Aplica filtros de búsqueda
     */
    private function applySearchFilters($query, $request)
    {
        if ($request->has('search') && $request->input('search') !== null) {
            $searchTerm = trim($request->input('search'));
            if (!empty($searchTerm)) {
                $query->where(function ($q) use ($searchTerm) {
                    $q->where('nombre_cliente', 'LIKE', "%{$searchTerm}%")
                        ->orWhere('numero_documento', 'LIKE', "%{$searchTerm}%")
                        ->orWhere('telefono_principal', 'LIKE', "%{$searchTerm}%")
                        ->orWhere('telefono_adicional', 'LIKE', "%{$searchTerm}%")
                        ->orWhere('correo_referencia', 'LIKE', "%{$searchTerm}%")
                        ->orWhere('orden_trabajo_anterior', 'LIKE', "%{$searchTerm}%")
                        ->orWhere('direccion_historico', 'LIKE', "%{$searchTerm}%")
                        ->orWhere('observaciones', 'LIKE', "%{$searchTerm}%")
                        ->orWhere('marca_base', 'LIKE', "%{$searchTerm}%")
                        ->orWhere('origen_base', 'LIKE', "%{$searchTerm}%")
                        ->orWhere('origen_motivo_cancelacion', 'LIKE', "%{$searchTerm}%");
                });
            }
        }
    }

    /**
     * Aplica filtros de trazabilidad
     */
    private function applyTraceabilityFilters($query, $request)
    {
        if ($request->filled('trazabilidad')) {
            $trazabilidad = $request->input('trazabilidad');
            if (is_array($trazabilidad)) {
                $query->whereIn('trazabilidad', $trazabilidad);
            } else {
                $query->where('trazabilidad', $trazabilidad);
            }
        }
    }

    /**
     * Aplica filtros directos como upload_id
     */
    private function applyDirectFilters($query, $request)
    {
        // Filtro directo por upload_id
        if ($request->filled('upload_id')) {
            $query->where('upload_id', $request->input('upload_id'));
        }
    }

    /**
     * Aplica filtros según permisos del usuario
     */
    private function applyPermissionFilters($query, $user, $desde, $hasta)
    {
        if ($user->can('vodafone.filtrar')) {
            $query->whereBetween('created_at', [$desde, $hasta]);
            if (!$user->can('vodafone.ver-global')) {
                $query->where('asignado_a_id', $user->id);
            }
        } elseif ($user->can('vodafone.recibe-asignacion') && !$user->can('vodafone.ver-global')) {
            $hoy = Carbon::today()->startOfDay();
            $finDelDia = Carbon::today()->endOfDay();
            $query->where('asignado_a_id', $user->id)
                ->where('trazabilidad', 'asignado')
                ->whereBetween('created_at', [$hoy, $finDelDia]);
        } elseif ($user->can('vodafone.ver') && !$user->can('vodafone.ver-global') && !$user->can('vodafone.recibe-asignacion')) {
            $query->where('trazabilidad', 'completado')
                ->whereBetween('created_at', [$desde, $hasta]);
        } else {
            $query->whereBetween('created_at', [$desde, $hasta]);
            if (!$user->can('vodafone.ver-global')) {
                $query->where('user_id', $user->id);
            }
        }
    }

    /**
     * Obtiene datos paginados para el frontend
     */
    private function getPaginatedData($query, $request)
    {
        /** @var \App\Models\User $user */
        $user = Auth::user();

        // Parámetros de paginación
        $page = max(1, (int) $request->input('page', 1));
        $perPage = max(10, min(100, (int) $request->input('per_page', 50)));
        $viewMode = $request->input('view_mode', 'grid');

        // Contar total de registros
        $totalRecords = $query->count();

        // Calcular offset
        $offset = ($page - 1) * $perPage;

        // Aplicar ordenamiento según el rol del usuario
        $queryWithOrder = $this->applyOrdering($query, $user);

        // Obtener registros paginados
        $items = $queryWithOrder
            ->offset($offset)
            ->limit($perPage)
            ->get();

        // Formatear datos
        $formattedItems = $this->formatItems($items);

        // Calcular información de paginación
        $totalPages = (int) ceil($totalRecords / $perPage);

        return [
            'data' => $formattedItems,
            'pagination' => [
                'current_page' => $page,
                'per_page' => $perPage,
                'total_records' => $totalRecords,
                'total_pages' => $totalPages,
                'has_next_page' => $page < $totalPages,
                'has_prev_page' => $page > 1,
                'from' => min($offset + 1, $totalRecords),
                'to' => min($offset + $perPage, $totalRecords),
            ],
            'view_mode' => $viewMode,
        ];
    }

    /**
     * Aplica ordenamiento según el rol del usuario
     */
    private function applyOrdering($query, $user)
    {
        if ($user->can('vodafone.ver') && !$user->can('vodafone.ver-global') && !$user->can('vodafone.recibe-asignacion')) {
            return $query->orderByRaw('RAND()');
        }

        return $query->orderBy('id');
    }

    /**
     * Formatea los items para el frontend
     */
    private function formatItems($items)
    {
        $hoy = Carbon::today();
        $ayer = Carbon::yesterday();

        return $items->transform(function ($item) use ($hoy, $ayer) {
            // Formato de fecha amigable
            $fecha = Carbon::parse($item->created_at);
            $dia = $fecha->isSameDay($hoy) ? 'Hoy' : ($fecha->isSameDay($ayer) ? 'Ayer' : $fecha->format('d/m/Y'));
            $hora = $fecha->format('g:i A');
            $item->created_at_formatted = "$dia $hora";

            // Cargar historial de asignaciones
            $item->asignaciones_historial = $this->getAssignmentHistory($item->id);
            $ultimaAsignacion = $item->asignaciones_historial->first();
            $item->ultima_asignacion = $ultimaAsignacion;
            $item->auditoria_historial = $ultimaAsignacion ? $ultimaAsignacion->auditoria_historial : collect();

            // Cargar relación tipo_documento (puede ser null)
            $tipo = null;
            if (isset($item->tipoDocumento) && $item->tipoDocumento) {
                $tipo = $item->tipoDocumento->codigo;
                $item->tipo_documento = $item->tipoDocumento; // para compatibilidad frontend
            }
            $label = $item->numero_documento ?? '';
            $item->numero_documento_con_tipo = $tipo ? ($tipo . ': ' . $label) : $label;

            return $item;
        });
    }


    private function getAssignableUsers()
    {
        return User::permission('vodafone.recibe-asignacion')
            ->get();
    }

    // =======================
    // MÉTODOS AUXILIARES PARA HISTORIAL Y AUDITORÍA
    // =======================

    private function getAssignmentHistory($vodafoneId)
    {
        $asignaciones = VodafoneAsignacion::with(['asignadoDe', 'asignadoA', 'usuarioCambio'])
            ->where('vodafone_id', $vodafoneId)
            ->orderByDesc('fecha')
            ->get();

        // Agregar historial de auditoría a cada cabecera
        foreach ($asignaciones as $cabecera) {
            $cabecera->auditoria_historial = $this->getAuditHistory($cabecera->id);
        }

        return $asignaciones;
    }

    private function getAuditHistory($asignacionId)
    {
        return VodafoneAuditoria::with('usuario')
            ->where('asignacion_id', $asignacionId)
            ->orderByDesc('fecha')
            ->get()
            ->map(function ($auditoria) {
                $campos = $auditoria->campos_editados;
                if (is_array($campos)) {
                    unset($campos['asignado_a_id']);
                    $auditoria->setAttribute('campos_editados', $campos);
                }
                $auditoria->hasRelevantChanges = is_array($auditoria->campos_editados) && count($auditoria->campos_editados) > 0;
                return $auditoria;
            })
            ->filter(function ($auditoria) {
                return $auditoria->hasRelevantChanges;
            })
            ->values();
    }

    // =======================
    // MÉTODOS AUXILIARES PARA VALIDACIÓN Y PROCESAMIENTO
    // =======================

    /**
     * Valida permisos de edición
     */
    private function validateEditPermissions($vodafone, $user, $data)
    {
        if ($vodafone->trazabilidad === 'completado') {
            if ($user->can('vodafone.editar-completados')) {
                return true;
            } else {
                return redirect()->back()->withErrors([
                    'general' => 'Este registro ya está completado y no puede ser editado.'
                ]);
            }
        }

        if (!in_array($vodafone->trazabilidad, ['asignado']) && $vodafone->asignado_a_id === $user->id) {
            return redirect()->back()->withErrors([
                'general' => 'El registro ya no está disponible para edición. Actualiza la página.'
            ]);
        }

        return true;
    }

    /**
     * Detecta cambios en los datos
     */
    private function detectChanges($vodafone, $data)
    {
        $camposEditados = [];
        $cambios = [];
        $campos = [
            'orden_trabajo_anterior',
            'origen_base',
            'marca_base',
            'origen_motivo_cancelacion',
            'nombre_cliente',
            'tipo_documento_id',
            'numero_documento',
            'telefono_principal',
            'telefono_adicional',
            'correo_referencia',
            'direccion_historico',
            'observaciones',
            'trazabilidad',
        ];

        foreach ($campos as $campo) {
            if (array_key_exists($campo, $data)) {
                $old = $vodafone->$campo;
                $new = $data[$campo];

                $oldNormalized = $old === null ? '' : (string)$old;
                $newNormalized = $new === null ? '' : (string)$new;

                if ($oldNormalized !== $newNormalized) {
                    $camposEditados[] = $campo;
                    $cambios[$campo] = ['old' => $old, 'new' => $new];
                }
            }
        }

        return ['campos' => $camposEditados, 'cambios' => $cambios];
    }

    /**
     * Auto-completa si está listo
     */
    private function autoCompleteIfReady($data)
    {
        $camposRequeridos = [
            'orden_trabajo_anterior',
            'origen_base',
            'marca_base',
            'origen_motivo_cancelacion',
            'nombre_cliente',
            'tipo_documento_id',
            'numero_documento',
            'telefono_principal',
            'telefono_adicional',
            'correo_referencia',
            'direccion_historico',
            'observaciones',
        ];

        $todosCompletos = true;
        foreach ($camposRequeridos as $campo) {
            if (empty($data[$campo])) {
                $todosCompletos = false;
                break;
            }
        }

        if ($todosCompletos) {
            $data['trazabilidad'] = 'completado';
        }

        return $data;
    }

    /**
     * Procesa las asignaciones
     */
    private function processAssignments($ids, $asignadoA, $user)
    {
        $cantidad = 0;

        foreach ($ids as $id) {
            $vodafone = Vodafone::find($id);
            if (!$vodafone) continue;

            $asignadoDe = $vodafone->asignado_a_id;

            if ($asignadoDe != $asignadoA) {
                $this->executeAssignment($vodafone, $asignadoDe, $asignadoA, $user);
                $cantidad++;
            }
        }

        return $cantidad;
    }

    /**
     * Ejecuta una asignación individual
     */
    private function executeAssignment($vodafone, $asignadoDe, $asignadoA, $user)
    {
        $vodafone->update([
            'asignado_a_id' => $asignadoA,
            'trazabilidad' => 'asignado',
            'updated_at' => now(),
        ]);

        $asignacion = VodafoneAsignacion::create([
            'vodafone_id' => $vodafone->id,
            'asignado_de_id' => $asignadoDe,
            'asignado_a_id' => $asignadoA,
            'user_id' => $user->id,
            'motivo' => 'reasignacion',
            'fecha' => now(),
        ]);

        VodafoneAuditoria::create([
            'vodafone_id' => $vodafone->id,
            'asignacion_id' => $asignacion->id,
            'user_id' => $user->id,
            'accion' => 'asignacion',
            'campos_editados' => [
                'asignado_a_id' => ['old' => $asignadoDe, 'new' => $asignadoA]
            ],
            'fecha' => now(),
        ]);
    }

    /**
     * Guarda pista de auditoría para cambios
     */
    private function saveAuditTrail($vodafone, $user, $cambios)
    {
        $ultimaAsignacion = VodafoneAsignacion::where('vodafone_id', $vodafone->id)
            ->orderByDesc('fecha')
            ->first();

        return VodafoneAuditoria::create([
            'vodafone_id' => $vodafone->id,
            'asignacion_id' => $ultimaAsignacion ? $ultimaAsignacion->id : null,
            'user_id' => $user->id,
            'accion' => 'edicion',
            'campos_editados' => $cambios,
            'fecha' => now(),
        ]);
    }

    /**
     * Guarda pista de auditoría para agendamiento
     */
    private function saveScheduleAuditTrail($vodafone, $user, $trazabilidadAnterior)
    {
        $ultimaAsignacion = VodafoneAsignacion::where('vodafone_id', $vodafone->id)
            ->orderByDesc('fecha')
            ->first();

        VodafoneAuditoria::create([
            'vodafone_id' => $vodafone->id,
            'asignacion_id' => $ultimaAsignacion ? $ultimaAsignacion->id : null,
            'user_id' => $user->id,
            'accion' => 'agendado',
            'campos_editados' => [
                'trazabilidad' => [
                    'anterior' => $trazabilidadAnterior,
                    'nueva' => 'agendado'
                ]
            ],
            'fecha' => now(),
        ]);
    }

    private function logAssignmentActivity($ids, $asignadoA, $userId, $cantidad)
    {
        // Log activity without detailed logging
    }

    // =======================
    // REGLAS DE VALIDACIÓN
    // =======================

    private function validationRules(?int $ignoreId = null): array
    {
        return [
            'trazabilidad' => ['nullable', 'in:pendiente,asignado,irrelevante,completado,retornado'],
            'orden_trabajo_anterior' => 'nullable|string|max:255',
            'origen_base' => 'nullable|string|max:255',
            'nombre_cliente' => 'required|string|max:255',
            'tipo_documento_id' => 'nullable|exists:tipos_documento,id',
            'numero_documento' => ['required', 'string', 'max:255', 'unique:historial_registros_vodafone,numero_documento,' . $ignoreId],
            'telefono_principal' => 'required|string|max:20',
            'telefono_adicional' => 'nullable|string|max:20',
            'correo_referencia' => 'nullable|email|max:255',
            'direccion_historico' => 'nullable|string|max:255',
            'marca_base' => 'nullable|string|max:255',
            'origen_motivo_cancelacion' => 'nullable|string|max:255',
            'observaciones' => 'nullable|string',
            'asignado_a_id' => 'nullable|exists:users,id',

            // Nuevos campos operador y whatsapp
            'operador_telefono_principal' => 'nullable|string|max:255',
            'ultact_operador_tel_prin' => 'nullable|date',
            'operador_telefono_adicional' => 'nullable|string|max:255',
            'ultact_operador_tel_adic' => 'nullable|date',
            'operador_telefono_prin_whatsapp' => 'nullable|boolean',
            'operador_telefono_adic_whatsapp' => 'nullable|boolean',

            // Nuevos campos tipo telefono
            'tipo_telefono_principal' => 'nullable|string|max:50',
            'tipo_telefono_adicional' => 'nullable|string|max:50',

            // Nuevos campos tipo de WhatsApp
            'tipo_whatsapp_principal' => 'nullable|string|in:business,normal',
            'tipo_whatsapp_adicional' => 'nullable|string|in:business,normal',
        ];
    }

    // =======================
    // MÉTODOS DE PLANTILLA
    // =======================

    public function descargarPlantilla()
    {
        try {
            // Crear un nuevo archivo Excel
            $spreadsheet = new \PhpOffice\PhpSpreadsheet\Spreadsheet();
            $sheet = $spreadsheet->getActiveSheet();

            // Establecer el título de la hoja
            $sheet->setTitle('Plantilla Vodafone');

            // Cabeceras del archivo
            $headers = [
                'orden_trabajo_anterior',
                'origen_base',
                'nombre_cliente',
                'numero_documento',
                'telefono_principal',
                'telefono_adicional',
                'correo_referencia',
                'direccion_historico',
                'marca_base',
                'origen_motivo_cancelacion',
                'observaciones'
            ];

            // Escribir cabeceras en la primera fila
            foreach ($headers as $index => $header) {
                $columnLetter = \PhpOffice\PhpSpreadsheet\Cell\Coordinate::stringFromColumnIndex($index + 1);
                $sheet->setCellValue($columnLetter . '1', $header);
            }

            // Aplicar formato a las cabeceras
            $lastColumn = \PhpOffice\PhpSpreadsheet\Cell\Coordinate::stringFromColumnIndex(count($headers));
            $sheet->getStyle('A1:' . $lastColumn . '1')->applyFromArray([
                'font' => [
                    'bold' => true,
                    'color' => ['rgb' => '000000'],
                ],
                'fill' => [
                    'fillType' => \PhpOffice\PhpSpreadsheet\Style\Fill::FILL_SOLID,
                    'startColor' => ['rgb' => 'E3F2FD'],
                ],
                'borders' => [
                    'allBorders' => [
                        'borderStyle' => \PhpOffice\PhpSpreadsheet\Style\Border::BORDER_THIN,
                        'color' => ['rgb' => '000000'],
                    ],
                ],
            ]);

            // Datos de ejemplo SOLO para los campos solicitados
            $ejemplo = [
                'orden_trabajo_anterior' => 'OT123456',
                'origen_base' => 'XXXXXXXXXX',
                'nombre_cliente' => '',
                'numero_documento' => '12345678X',
                'telefono_principal' => '600123456',
                'telefono_adicional' => '600654321',
                'correo_referencia' => 'example@xxxxx.com',
                'direccion_historico' => 'Av.Tacna 123, Lima-Perú',
                'marca_base' => '',
                'origen_motivo_cancelacion' => '',
                'observaciones' => ''
            ];


            // Escribir datos de ejemplo en la fila 2
            foreach ($headers as $colIndex => $header) {
                $columnLetter = \PhpOffice\PhpSpreadsheet\Cell\Coordinate::stringFromColumnIndex($colIndex + 1);
                $sheet->setCellValue($columnLetter . '2', $ejemplo[$header] ?? '');
            }

            // Ajustar el ancho de las columnas automáticamente
            foreach (range(1, count($headers)) as $colIndex) {
                $columnLetter = \PhpOffice\PhpSpreadsheet\Cell\Coordinate::stringFromColumnIndex($colIndex);
                $sheet->getColumnDimension($columnLetter)->setAutoSize(true);
            }

            // Crear el writer para Excel
            $writer = new \PhpOffice\PhpSpreadsheet\Writer\Xlsx($spreadsheet);

            // Crear el nombre del archivo
            $filename = 'plantilla_vodafone_' . date('Y-m-d') . '.xlsx';

            // Crear respuesta con el archivo Excel
            return response()->streamDownload(function () use ($writer) {
                $writer->save('php://output');
            }, $filename, [
                'Content-Type' => 'application/vnd.openxmlformats-officedocument.spreadsheetml.sheet',
                'Cache-Control' => 'no-cache, no-store, must-revalidate',
                'Pragma' => 'no-cache',
                'Expires' => '0',
            ]);
        } catch (\Exception $e) {
            return redirect()->back()->with('error', 'Error al generar la plantilla Excel: ' . $e->getMessage());
        }
    }
}
