<?php

namespace App\Jobs;

use App\Models\Vodafone;
use App\Models\LogImportacionVodafone;
use App\Models\User;
use Illuminate\Bus\Queueable;
use Illuminate\Contracts\Queue\ShouldQueue;
use Illuminate\Queue\InteractsWithQueue;
use Illuminate\Queue\SerializesModels;
use Illuminate\Support\Facades\Log;
use Illuminate\Support\Facades\DB;

class VodafoneImportJob implements ShouldQueue
{
    use InteractsWithQueue, Queueable, SerializesModels;

    protected $datos;
    protected $modo;
    protected $logId;
    protected $userId;

    public function __construct($datos, $modo, $logId, $userId)
    {
        $this->datos = $datos;
        $this->modo = $modo;
        $this->logId = $logId;
        $this->userId = $userId;
    }

    public function handle()
    {
        $errores = [];
        $user = User::find($this->userId);
        $log = LogImportacionVodafone::find($this->logId);

        $modo = strtolower(trim($this->modo ?? ''));

        Log::info("🚀 INICIANDO JOB DE IMPORTACIÓN", [
            'modo' => $modo,
            'log_id' => $this->logId,
            'user_id' => $this->userId,
            'cantidad_datos_recibidos' => count($this->datos),
            'primer_dato_estructura' => array_keys($this->datos[0] ?? []),
            'primer_dato_completo' => $this->datos[0] ?? null,
        ]);


        if ($log) {
            $log->estado = 'procesando'; // 🔥 ESTADO EN PROCESO
            $log->save();
        }

        DB::beginTransaction();
        try {
            foreach ($this->datos as $row) {
                $documento = trim($row['numero_documento'] ?? ''); // ✅ CORREGIDO: era dni_cliente
                $telefono = trim($row['telefono_principal'] ?? '');
                $nombre = trim($row['nombre_cliente'] ?? '');

                // Validar que al menos tenga uno de los campos clave
                if (!$documento && !$telefono && !$nombre) {
                    Log::info("Fila omitida: sin datos mínimos", ['row' => $row]);
                    continue;
                }

                $data = [
                    'orden_trabajo_anterior' => $row['orden_trabajo_anterior'] ?? null,
                    'origen_base' => $row['origen_base'] ?? null,
                    'nombre_cliente' => $nombre ?: null,
                    'numero_documento' => $documento ?: null,
                    'tipo_documento_id' => $row['tipo_documento_id'] ?? null, // ✅ NUEVO: Tipo detectado automáticamente
                    'telefono_principal' => $telefono ?: null,
                    'telefono_adicional' => $row['telefono_adicional'] ?? null,
                    'correo_referencia' => $row['correo_referencia'] ?? null,
                    'direccion_historico' => $row['direccion_historico'] ?? null,
                    'marca_base' => $row['marca_base'] ?? null,
                    'origen_motivo_cancelacion' => $row['origen_motivo_cancelacion'] ?? null,
                    'observaciones' => $row['observaciones'] ?? null,
                    'trazabilidad' => 'pendiente',
                    'asignado_a_id' => $row['asignado_a_id'] ?? null,
                    'user_id' => $this->userId,
                    'upload_id' => $this->logId,
                ];

                try {
                    Log::info("Procesando fila", [
                        'numero_documento' => $documento,
                        'telefono_principal' => $telefono,
                        'nombre_cliente' => $nombre
                    ]);

                    // Buscar por documento o teléfono
                    $existente = null;
                    if ($documento) {
                        $existente = Vodafone::withTrashed()
                            ->where('numero_documento', $documento) // ✅ CORREGIDO: era dni_cliente
                            ->first();
                    }
                    if (!$existente && $telefono) {
                        $existente = Vodafone::withTrashed()
                            ->where('telefono_principal', $telefono)
                            ->first();
                    }

                    if ($existente) {
                        Log::info("Registro encontrado", [
                            'numero_documento' => $documento,
                            'telefono_principal' => $telefono,
                            'id' => $existente->id,
                            'eliminado_logicamente' => $existente->trashed()
                        ]);

                        if ($modo === 'actualizar') {
                            if ($existente->trashed()) {
                                Log::info("Restaurando registro eliminado", ['numero_documento' => $documento]);
                                $existente->restore();
                            }
                            $existente->update($data);
                            Log::info("Registro actualizado", ['numero_documento' => $documento]);
                        } elseif ($modo === 'omitir') {
                            // ✅ CORREGIDO: Si está eliminado lógicamente, restaurarlo
                            if ($existente->trashed()) {
                                Log::info("Restaurando registro eliminado (modo omitir)", ['numero_documento' => $documento]);
                                $existente->restore();
                                $existente->update($data);
                                Log::info("Registro restaurado y actualizado", ['numero_documento' => $documento]);
                            } else {
                                Log::info("Registro omitido - ya existe activo", ['numero_documento' => $documento]);
                            }
                        } else {
                            // Modo por defecto: reemplazar completamente
                            Log::info("Reemplazando registro existente", ['numero_documento' => $documento]);
                            $existente->forceDelete();
                            $nuevo = Vodafone::create($data);
                            Log::info("Registro recreado tras eliminación", ['numero_documento' => $documento, 'nuevo_id' => $nuevo->id]);
                        }
                    } else {
                        Log::info("No se encontró registro existente. Creando nuevo", ['numero_documento' => $documento]);
                        $nuevo = Vodafone::create($data);
                        Log::info("Registro creado exitosamente", ['numero_documento' => $documento, 'nuevo_id' => $nuevo->id]);
                    }
                } catch (\Throwable $e) {
                    $identificador = $documento ?: $telefono ?: $nombre;
                    $errores[] = "Error con registro {$identificador}: " . $e->getMessage();
                    Log::error("Error con registro {$identificador}: " . $e->getMessage());
                }
            }

            DB::commit();
        } catch (\Throwable $e) {
            DB::rollBack();
            $errores[] = "Error general: " . $e->getMessage();
            Log::error('Error en importación masiva Vodafone: ' . $e->getMessage());
        }

        if ($log) {
            $log->estado = 'finalizado'; // 🏁 FINALIZADO
            $log->errores_json = json_encode($errores);
            $log->save();
        }
    }
}
