<?php

namespace App\Models;

use Illuminate\Database\Eloquent\Factories\HasFactory;
use Illuminate\Database\Eloquent\Model;
use Illuminate\Database\Eloquent\Relations\BelongsTo;
use Illuminate\Database\Eloquent\Relations\HasMany;

class EnriquecimientoImportacion extends Model
{
    use HasFactory;

    protected $table = 'enriquecimiento_importaciones';

    protected $fillable = [
        'user_id',
        'nombre_archivo',
        'tipo_importacion',
        'total_numeros',
        'numeros_procesados',
        'numeros_pendientes',
        'creditos_consumidos',
        'estado',
        'metadatos',
        'observaciones'
    ];

    protected $casts = [
        'metadatos' => 'array',
        'total_numeros' => 'integer',
        'numeros_procesados' => 'integer',
        'numeros_pendientes' => 'integer',
        'creditos_consumidos' => 'integer'
    ];

    /**
     * Relación con el usuario que creó la importación
     */
    public function usuario(): BelongsTo
    {
        return $this->belongsTo(User::class, 'user_id');
    }

    /**
     * Relación con los números de esta importación
     */
    public function numeros(): HasMany
    {
        return $this->hasMany(EnriquecimientoNumero::class, 'importacion_id');
    }

    /**
     * Scope para filtrar por estado
     */
    public function scopeByEstado($query, $estado)
    {
        return $query->where('estado', $estado);
    }

    /**
     * Scope para filtrar por usuario
     */
    public function scopeByUsuario($query, $userId)
    {
        return $query->where('user_id', $userId);
    }

    /**
     * Calcular progreso de la importación
     */
    public function getProgresoAttribute()
    {
        if ($this->total_numeros == 0) {
            return 0;
        }

        return round(($this->numeros_procesados / $this->total_numeros) * 100, 2);
    }

    /**
     * Verificar si la importación está completa
     */
    public function getEsCompletaAttribute()
    {
        return $this->estado === 'completada' && $this->numeros_procesados === $this->total_numeros;
    }
}
