<?php

namespace App\Models;

use Illuminate\Database\Eloquent\Model;
use Carbon\Carbon;

class UserSession extends Model
{
    protected $fillable = [
        'user_id',
        'session_id',
        'ip_address',
        'user_agent',
        'device_name',
        'location',
        'last_activity',
        'is_current',
    ];

    protected $casts = [
        'last_activity' => 'datetime',
        'is_current' => 'boolean',
    ];

    /**
     * Usuario propietario de la sesión
     */
    public function user()
    {
        return $this->belongsTo(User::class);
    }

    /**
     * Scope para sesiones activas (últimos 30 minutos)
     */
    public function scopeActivas($query)
    {
        return $query->where('last_activity', '>=', Carbon::now()->subMinutes(30));
    }

    /**
     * Scope para una sesión específica
     */
    public function scopePorSessionId($query, $sessionId)
    {
        return $query->where('session_id', $sessionId);
    }

    /**
     * Obtiene el nombre del dispositivo de forma amigable
     */
    public function getDeviceNameAttribute($value)
    {
        if ($value) {
            return $value;
        }

        // Extraer información del user agent
        $userAgent = $this->user_agent;

        if (strpos($userAgent, 'Chrome') !== false) {
            return 'Navegador Chrome';
        } elseif (strpos($userAgent, 'Firefox') !== false) {
            return 'Navegador Firefox';
        } elseif (strpos($userAgent, 'Safari') !== false) {
            return 'Navegador Safari';
        } elseif (strpos($userAgent, 'Edge') !== false) {
            return 'Navegador Edge';
        }

        return 'Navegador desconocido';
    }

    /**
     * Verifica si la sesión está activa
     */
    public function isActive()
    {
        return $this->last_activity >= Carbon::now()->subMinutes(30);
    }

    /**
     * Marca esta sesión como la actual
     */
    public function markAsCurrent()
    {
        // Desmarcar todas las demás sesiones del usuario
        static::where('user_id', $this->user_id)->update(['is_current' => false]);

        // Marcar esta como actual
        $this->update(['is_current' => true]);
    }

    /**
     * Eliminar sesiones inactivas
     */
    public static function limpiarSesionesInactivas()
    {
        return static::where('last_activity', '<', Carbon::now()->subMinutes(30))->delete();
    }
}
