<?php

namespace App\Models;

use Illuminate\Database\Eloquent\Factories\HasFactory;
use Illuminate\Database\Eloquent\Model;
use Illuminate\Database\Eloquent\Relations\BelongsTo;
use Carbon\Carbon;

class UsuarioCredito extends Model
{
    use HasFactory;

    protected $table = 'usuario_creditos';

    protected $fillable = [
        'user_id',
        'fecha_periodo',
        'tipo_periodo',
        'creditos_utilizados',
        'limite_creditos',
        'ultimo_uso',
        'detalle_uso'
    ];

    protected $casts = [
        'fecha_periodo' => 'date',
        'creditos_utilizados' => 'integer',
        'limite_creditos' => 'integer',
        'ultimo_uso' => 'datetime',
        'detalle_uso' => 'array'
    ];

    /**
     * Relación con el usuario
     */
    public function usuario(): BelongsTo
    {
        return $this->belongsTo(User::class, 'user_id');
    }

    /**
     * MÉTODO PRINCIPAL: Verificar si el usuario puede usar créditos
     * Verifica tanto créditos diarios como mensuales
     */
    public static function puedeUsarCreditos($userId, $cantidad, $tipoPeriodo = null)
    {
        // Si no se especifica tipo, verificamos ambos (diario Y mensual)
        if ($tipoPeriodo === null) {
            return self::puedeUsarCreditos($userId, $cantidad, 'diario') &&
                self::puedeUsarCreditos($userId, $cantidad, 'mensual');
        }

        $fechaPeriodo = $tipoPeriodo === 'diario' ? Carbon::today() : Carbon::now()->startOfMonth();

        $credito = self::where('user_id', $userId)
            ->where('fecha_periodo', $fechaPeriodo)
            ->where('tipo_periodo', $tipoPeriodo)
            ->first();

        if (!$credito) {
            // Si no existe el registro, obtener límite del plan del usuario
            $limite = self::obtenerLimiteCreditos($userId, $tipoPeriodo);
            return $cantidad <= $limite;
        }

        return ($credito->creditos_utilizados + $cantidad) <= $credito->limite_creditos;
    }

    /**
     * MÉTODO PRINCIPAL: Consumir créditos después de procesar números
     * Consume tanto créditos diarios como mensuales
     */
    public static function consumirCreditos($userId, $cantidad, $detalle = null)
    {
        // Consumir créditos diarios
        $creditoDiario = self::consumirCreditosPorTipo($userId, $cantidad, $detalle, 'diario');

        // Consumir créditos mensuales
        $creditoMensual = self::consumirCreditosPorTipo($userId, $cantidad, $detalle, 'mensual');

        return [
            'diario' => $creditoDiario,
            'mensual' => $creditoMensual
        ];
    }

    /**
     * Consumir créditos de un tipo específico
     */
    private static function consumirCreditosPorTipo($userId, $cantidad, $detalle, $tipoPeriodo)
    {
        $fechaPeriodo = $tipoPeriodo === 'diario' ? Carbon::today() : Carbon::now()->startOfMonth();

        // Buscar o crear el registro
        $credito = self::firstOrCreate(
            [
                'user_id' => $userId,
                'fecha_periodo' => $fechaPeriodo,
                'tipo_periodo' => $tipoPeriodo
            ],
            [
                'creditos_utilizados' => 0,
                'limite_creditos' => self::obtenerLimiteCreditos($userId, $tipoPeriodo)
            ]
        );

        // Incrementar directamente sin usar increment() para evitar problemas de concurrencia
        $credito->creditos_utilizados += $cantidad;
        $credito->ultimo_uso = now();
        $credito->detalle_uso = array_merge($credito->detalle_uso ?? [], [$detalle]);
        $credito->save();

        return $credito;
    }

    /**
     * Verificar cuántos números se pueden procesar con créditos disponibles
     */
    public static function numerosDisponiblesParaProcesar($userId)
    {
        $creditosDisponiblesDiarios = self::getCreditosDisponibles($userId, 'diario');
        $creditosDisponiblesMensuales = self::getCreditosDisponibles($userId, 'mensual');

        // El límite es el menor entre diarios y mensuales
        return min($creditosDisponiblesDiarios, $creditosDisponiblesMensuales);
    }

    /**
     * Obtener créditos disponibles por tipo
     */
    public static function getCreditosDisponibles($userId, $tipoPeriodo)
    {
        $fechaPeriodo = $tipoPeriodo === 'diario' ? Carbon::today() : Carbon::now()->startOfMonth();

        $credito = self::where('user_id', $userId)
            ->where('fecha_periodo', $fechaPeriodo)
            ->where('tipo_periodo', $tipoPeriodo)
            ->first();

        $limite = self::obtenerLimiteCreditos($userId, $tipoPeriodo);
        $usados = $credito ? $credito->creditos_utilizados : 0;

        return max(0, $limite - $usados);
    }

    /**
     * Obtener límite de créditos según plan del usuario
     */
    private static function obtenerLimiteCreditos($userId, $tipoPeriodo)
    {
        $usuario = User::with('plan')->find($userId);

        return $tipoPeriodo === 'diario'
            ? $usuario->plan->creditos_diarios ?? 0
            : $usuario->plan->creditos_mensuales ?? 0;
    }

    /**
     * Obtener créditos disponibles
     */
    public function getCreditosDisponiblesAttribute()
    {
        return $this->limite_creditos - $this->creditos_utilizados;
    }

    /**
     * Scope para período actual
     */
    public function scopePeriodoActual($query, $tipoPeriodo = 'diario')
    {
        $fechaPeriodo = $tipoPeriodo === 'diario' ? Carbon::today() : Carbon::now()->startOfMonth();

        return $query->where('fecha_periodo', $fechaPeriodo)
            ->where('tipo_periodo', $tipoPeriodo);
    }

    /**
     * Obtener estadísticas de uso para el dashboard
     */
    public static function getEstadisticasUsuario($userId)
    {
        $hoy = Carbon::today();
        $mesActual = Carbon::now()->startOfMonth();

        $creditosHoy = self::where('user_id', $userId)
            ->where('fecha_periodo', $hoy)
            ->where('tipo_periodo', 'diario')
            ->first();

        $creditosMes = self::where('user_id', $userId)
            ->where('fecha_periodo', $mesActual)
            ->where('tipo_periodo', 'mensual')
            ->first();

        $usuario = User::with('plan')->find($userId);

        return [
            'diarios' => [
                'usados' => $creditosHoy ? $creditosHoy->creditos_utilizados : 0,
                'limite' => $usuario->plan->creditos_diarios ?? 0,
                'disponibles' => ($usuario->plan->creditos_diarios ?? 0) - ($creditosHoy ? $creditosHoy->creditos_utilizados : 0)
            ],
            'mensuales' => [
                'usados' => $creditosMes ? $creditosMes->creditos_utilizados : 0,
                'limite' => $usuario->plan->creditos_mensuales ?? 0,
                'disponibles' => ($usuario->plan->creditos_mensuales ?? 0) - ($creditosMes ? $creditosMes->creditos_utilizados : 0)
            ]
        ];
    }

    /**
     * Obtener historial detallado de consumo de créditos
     */
    public static function getHistorialDetallado($userId, $limite = 50)
    {
        return self::where('user_id', $userId)
            ->whereNotNull('detalle_uso')
            ->orderBy('updated_at', 'desc')
            ->limit($limite)
            ->get()
            ->map(function ($credito) {
                $detalles = [];

                foreach (($credito->detalle_uso ?? []) as $detalle) {
                    if (is_array($detalle)) {
                        $detalles[] = [
                            'descripcion' => $detalle['tipo'] ?? 'Uso de crédito',
                            'fecha' => $detalle['fecha'] ?? $credito->updated_at->toDateTimeString(),
                            'numero' => $detalle['numero'] ?? null,
                            'importacion_id' => $detalle['importacion_id'] ?? null,
                            'tipo_periodo' => $credito->tipo_periodo,
                            'fecha_periodo' => $credito->fecha_periodo
                        ];
                    } else {
                        $detalles[] = [
                            'descripcion' => $detalle,
                            'fecha' => $credito->updated_at->toDateTimeString(),
                            'numero' => null,
                            'importacion_id' => null,
                            'tipo_periodo' => $credito->tipo_periodo,
                            'fecha_periodo' => $credito->fecha_periodo
                        ];
                    }
                }

                return $detalles;
            })
            ->flatten(1)
            ->take($limite)
            ->values();
    }
}
