<?php

namespace App\Models;

use Illuminate\Database\Eloquent\Model;
use Illuminate\Database\Eloquent\SoftDeletes;
use App\Models\User;

class Vodafone extends Model
{
    use SoftDeletes;

    protected $table = 'historial_registros_vodafone';

    protected $fillable = [
        'user_id',
        'upload_id',
        'asignado_a_id',
        'trazabilidad',
        'tipo_documento_id',

        // Nuevos campos de SmartClient / filtrado
        'orden_trabajo_anterior',
        'origen_base',
        'nombre_cliente',
        'numero_documento',
        'telefono_principal',
        'telefono_adicional',
        'correo_referencia',
        'direccion_historico',
        'marca_base',
        'origen_motivo_cancelacion',
        'observaciones',

        // Nuevos campos operador y whatsapp
        'operador_telefono_principal',
        'ultact_operador_tel_prin',
        'operador_telefono_adicional',
        'ultact_operador_tel_adic',
        'operador_telefono_prin_whatsapp',
        'operador_telefono_adic_whatsapp',

        // Nuevos campos tipo telefono
        'tipo_telefono_principal',
        'tipo_telefono_adicional',
    ];

    protected $casts = [
        'fecha_creacion' => 'date',
        'fecha_cierre' => 'date',
    ];

    public function user()
    {
        return $this->belongsTo(User::class);
    }

    public function logImportacion()
    {
        return $this->belongsTo(LogImportacionVodafone::class, 'upload_id');
    }

    public function creador()
    {
        return $this->belongsTo(User::class, 'user_id');
    }

    public function asignado_a()
    {
        return $this->belongsTo(User::class, 'asignado_a_id');
    }

    // Relación con TipoDocumento
    public function tipoDocumento()
    {
        return $this->belongsTo(TipoDocumento::class, 'tipo_documento_id');
    }

    // Accessor para obtener el documento completo del cliente
    public function getDocumentoClienteCompletoAttribute()
    {
        if (!$this->tipoDocumento || !$this->numero_documento) {
            return null;
        }
        return $this->tipoDocumento->codigo . ': ' . $this->numero_documento;
    }

    // Accessor para el frontend - numero_documento_con_tipo
    public function getNumeroDocumentoConTipoAttribute()
    {
        if (!$this->tipoDocumento || !$this->numero_documento) {
            return $this->numero_documento ?: '';
        }
        return $this->tipoDocumento->codigo . ': ' . $this->numero_documento;
    }

    // Scope para filtrar por tipo de documento
    public function scopePorTipoDocumento($query, $tipoDocumentoId)
    {
        return $query->where('tipo_documento_id', $tipoDocumentoId);
    }

    // Scope para filtrar por número de documento
    public function scopePorNumeroDocumento($query, $numeroDocumento)
    {
        return $query->where('numero_documento', 'LIKE', '%' . $numeroDocumento . '%');
    }
}
