import { ref, onMounted, onUnmounted } from "vue";
import { router } from "@inertiajs/vue3";
import Swal from "sweetalert2";
import axios from "axios";

export function useSessionControl() {
    let intervalId = null;
    let hasRedirected = false; // Flag para evitar redirecciones múltiples

    // Estados reactivos para compatibilidad con el layout
    const isSessionLimitExceeded = ref(false);
    const sessionData = ref(null);

    const checkSessions = async () => {
        try {
            const response = await axios.get("/sessions/status");
            const data = response.data;

            // Actualizar datos de sesión
            sessionData.value = data;

            // NUEVA LÓGICA: Solo actuar si ESTA sesión específica debe cerrarse
            if (data.limit_exceeded && data.should_close_this_session) {
                isSessionLimitExceeded.value = true;

                await Swal.fire({
                    title: "Sesión Cerrada",
                    html: `
                        <p>Tu sesión ha sido cerrada porque se ha excedido el límite de sesiones activas.</p>
                        <p>Plan actual: máximo ${data.sessions_limit} sesión(es) activa(s).</p>
                        <p>Las sesiones más antiguas se cierran automáticamente.</p>
                        <p>Serás redirigido al login automáticamente.</p>
                    `,
                    icon: "warning",
                    timer: 3000,
                    showConfirmButton: false,
                    allowOutsideClick: false,
                });

                // Esta sesión debe cerrar - hacer logout
                await axios.post("/logout");
                window.location.href = "/login";
            } else if (
                data.limit_exceeded &&
                data.is_newest_session &&
                !hasRedirected
            ) {
                // Esta es la sesión MÁS NUEVA que causó el exceso del límite
                // Redirigir a la página de manejo de sesiones para que el usuario decida
                hasRedirected = true; // Marcar para evitar redirecciones múltiples
                isSessionLimitExceeded.value = true;

                // Detener el monitoreo antes de redirigir
                stopMonitoring();

                // Redirigir inmediatamente
                window.location.href = "/session-limit";
            } else {
                isSessionLimitExceeded.value = false;
            }
        } catch (error) {
            // Error manejado silenciosamente en producción
        }
    };

    const startMonitoring = () => {
        // Para nuevas sesiones, verificar muy rápidamente
        const quickCheck = async () => {
            await checkSessions();

            // Si no hemos redirigido, hacer otra verificación rápida en 500ms
            if (!hasRedirected) {
                setTimeout(checkSessions, 500);
            }
        };

        // Verificar inmediatamente al cargar
        quickCheck();

        // Luego verificar cada 3 segundos
        intervalId = setInterval(() => {
            if (!hasRedirected) {
                checkSessions();
            }
        }, 3000);
    };

    const stopMonitoring = () => {
        if (intervalId) {
            clearInterval(intervalId);
            intervalId = null;
        }
    };

    onMounted(() => {
        startMonitoring();
    });

    onUnmounted(() => {
        stopMonitoring();
    });

    return {
        isSessionLimitExceeded,
        sessionData,
        checkSessions,
        checkSessionStatus: checkSessions, // Alias para compatibilidad
        startMonitoring,
        stopMonitoring,
    };
}
